/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.api.builder;

import static java.util.stream.Collectors.toList;

import org.mule.metadata.api.model.AttributeFieldType;
import org.mule.metadata.api.model.MetadataFormat;
import org.mule.metadata.api.model.ObjectKeyType;
import org.mule.metadata.api.model.impl.DefaultObjectKeyType;

import javax.xml.namespace.QName;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.regex.Pattern;

public class ObjectKeyBuilder extends AbstractBuilder<ObjectKeyType>
    implements TypeBuilder<ObjectKeyType> {

  private Optional<QName> name;
  private Optional<Pattern> pattern;
  private List<AttributeFieldTypeBuilder> attributes;

  public ObjectKeyBuilder(MetadataFormat format) {
    super(format);
    this.name = Optional.empty();
    this.pattern = Optional.empty();
    this.attributes = new ArrayList<>();
  }

  public ObjectKeyBuilder name(String name) {
    this.name = Optional.of(new QName(name));
    return this;
  }

  public ObjectKeyBuilder name(QName name) {
    this.name = Optional.of(name);
    return this;
  }

  public ObjectKeyBuilder pattern(Pattern name) {
    this.pattern = Optional.of(name);
    return this;
  }

  public AttributeFieldTypeBuilder addAttribute() {
    final AttributeFieldTypeBuilder attributeBuilder = new AttributeFieldTypeBuilder(format);
    attributes.add(attributeBuilder);
    return attributeBuilder;
  }

  @Override
  public ObjectKeyType build() {
    if (!name.isPresent() && !pattern.isPresent()) {
      throw new RuntimeException("No name or pattern was specified.");
    }

    final List<AttributeFieldType> attributes = this.attributes.stream()
        .map((builder) -> builder.build())
        .collect(toList());

    return new DefaultObjectKeyType(name, pattern, attributes, format, annotations);
  }
}
