/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.message.api.el;


import org.mule.metadata.api.model.MetadataType;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

public class ModuleDefinition {

  private ModuleIdentifier name;
  private Map<String, MetadataType> declaredElements;
  private List<MetadataType> declaredTypes;

  public ModuleDefinition(ModuleIdentifier name, Map<String, MetadataType> declaredElements) {
    this.name = name;
    this.declaredElements = declaredElements;
    this.declaredTypes = new ArrayList<>();
  }

  public ModuleDefinition(ModuleIdentifier name, Map<String, MetadataType> declaredElements, List<MetadataType> declaredTypes) {
    this.name = name;
    this.declaredElements = declaredElements;
    this.declaredTypes = declaredTypes;
  }

  public ModuleIdentifier getName() {
    return name;
  }

  public Collection<String> identifiers() {
    return declaredElements.keySet();
  }

  public List<MetadataType> declaredTypes() {
    return declaredTypes;
  }

  public Optional<MetadataType> lookup(String identifier) {
    return Optional.ofNullable(declaredElements.get(identifier));
  }

  public static Builder builder(String name) {
    return new Builder(name);
  }

  public static class Builder {

    private String name;
    private Map<String, MetadataType> declaredElements;
    private List<MetadataType> declaredTypes;

    public Builder(String name) {
      this.name = name;
      declaredElements = new HashMap<>();
      declaredTypes = new ArrayList<>();
    }

    public Builder addElement(String name, MetadataType type) {
      declaredElements.put(name, type);
      return this;
    }

    public Builder addType(MetadataType type) {
      declaredTypes.add(type);
      return this;
    }

    public ModuleDefinition build() {
      return new ModuleDefinition(ModuleIdentifier.fromString(name), declaredElements, declaredTypes);
    }
  }
}
