/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.message.api;


import org.mule.metadata.api.annotation.TypeAnnotation;
import org.mule.metadata.api.builder.ObjectFieldTypeBuilder;
import org.mule.metadata.api.model.MetadataFormat;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.api.model.ObjectFieldType;
import org.mule.metadata.api.model.ObjectType;
import org.mule.metadata.api.model.impl.BaseMetadataType;
import org.mule.metadata.api.visitor.MetadataTypeVisitor;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import static java.util.Optional.ofNullable;

/**
 * Default implementation for {@link MuleEventMetadataType}.
 *
 * @since 1.0
 */
class DefaultMuleEventMetadataType extends BaseMetadataType implements ObjectType, MuleEventMetadataType {

  public static final String MESSAGE_FIELD_NAME = "message";
  public static final String ERROR_FIELD_NAME = "error";
  public static final String VARIABLES_FIELD_NAME = "variables";
  private final ObjectFieldType messageMetadataType;
  private final ObjectFieldType variables;
  private final ObjectFieldType errorFieldType;


  DefaultMuleEventMetadataType(MessageMetadataType messageMetadataType, ObjectType variables,
                               Map<Class<? extends TypeAnnotation>, TypeAnnotation> annotations) {
    this(messageMetadataType, null, variables, annotations);
  }

  DefaultMuleEventMetadataType(MessageMetadataType messageMetadataType, MetadataType errorType, ObjectType variables,
                               Map<Class<? extends TypeAnnotation>, TypeAnnotation> annotations) {
    super(MetadataFormat.JAVA, annotations);
    final ObjectFieldTypeBuilder messageTypeBuilder = new ObjectFieldTypeBuilder(MetadataFormat.JAVA);
    messageTypeBuilder.key(MESSAGE_FIELD_NAME).value(messageMetadataType);
    messageTypeBuilder.required(true);
    this.messageMetadataType = messageTypeBuilder.build();

    if (errorType != null) {
      final ObjectFieldTypeBuilder errorTypeBuilder = new ObjectFieldTypeBuilder(MetadataFormat.JAVA);
      errorTypeBuilder.key(ERROR_FIELD_NAME).value(errorType);
      errorTypeBuilder.required(true);
      errorFieldType = errorTypeBuilder.build();
    } else {
      errorFieldType = null;
    }

    final ObjectFieldTypeBuilder variablesTypeBuilder = new ObjectFieldTypeBuilder(MetadataFormat.JAVA);
    variablesTypeBuilder.required(true);
    variablesTypeBuilder.key(VARIABLES_FIELD_NAME).value(variables);
    this.variables = variablesTypeBuilder.build();
  }

  private Optional<ObjectFieldType> getErrorFieldType() {
    return Optional.ofNullable(errorFieldType);
  }

  @Override
  public MessageMetadataType getMessageType() {
    return (MessageMetadataType) messageMetadataType.getValue();
  }

  @Override
  public Optional<MetadataType> getErrorType() {
    return getErrorFieldType().map(ObjectFieldType::getValue);
  }

  @Override
  public ObjectType getVariables() {
    return (ObjectType) variables.getValue();
  }

  @Override
  public Optional<MetadataType> getVariableType(String varName) {
    return getVariables().getFieldByName(varName).map(ObjectFieldType::getValue);
  }

  @Override
  public Optional<MetadataType> getOpenRestriction() {
    return Optional.empty();
  }

  @Override
  public Collection<ObjectFieldType> getFields() {
    final List<ObjectFieldType> objectFieldTypes = new ArrayList<>();
    objectFieldTypes.add(messageMetadataType);
    getErrorFieldType().ifPresent(objectFieldTypes::add);
    objectFieldTypes.add(variables);
    return objectFieldTypes;
  }

  @Override
  public boolean isOrdered() {
    return false;
  }

  @Override
  public Optional<ObjectFieldType> getFieldByName(String name) {
    switch (name) {
      case MESSAGE_FIELD_NAME:
        return ofNullable(messageMetadataType);
      case ERROR_FIELD_NAME:
        return getErrorFieldType();
      case VARIABLES_FIELD_NAME:
        return ofNullable(variables);
    }
    return Optional.empty();
  }

  @Override
  public void accept(MetadataTypeVisitor metadataTypeVisitor) {
    metadataTypeVisitor.visitObject(this);
  }
}
