/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.message.api.el;


import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.message.api.MessageMetadataType;
import org.mule.metadata.message.api.MuleEventMetadataType;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * Represents the definition of the a set of bindings to the expression language
 */
public class TypeBindings {

  public static final String PAYLOAD_IDENTIFIER_NAME = "payload";
  public static final String ATTRIBUTES_IDENTIFIER_NAME = "attributes";
  public static final String VARIABLES_IDENTIFIER_NAME = "vars";
  public static final String MESSAGE_IDENTIFIER_NAME = "message";

  private Map<String, MetadataType> bindings;
  private Collection<ModuleDefinition> modules;

  private TypeBindings(Map<String, MetadataType> bindings, Collection<ModuleDefinition> modules) {
    this.bindings = bindings;
    this.modules = modules;
  }

  public Set<String> identifiers() {
    return bindings.keySet();
  }

  public Optional<MetadataType> lookup(String name) {
    return Optional.ofNullable(bindings.get(name));
  }

  public Collection<ModuleDefinition> modules() {
    return modules;
  }

  public static Builder builder() {
    return new Builder();
  }

  public static Builder builder(MuleEventMetadataType muleEventMetadataType) {
    Builder builder = builder();
    MessageMetadataType messageType = muleEventMetadataType.getMessageType();
    messageType.getPayloadType().ifPresent(metadataType -> builder.addBinding(PAYLOAD_IDENTIFIER_NAME, metadataType));
    messageType.getAttributesType().ifPresent(metadataType -> builder.addBinding(ATTRIBUTES_IDENTIFIER_NAME, metadataType));
    builder.addBinding(VARIABLES_IDENTIFIER_NAME, muleEventMetadataType.getVariables());
    builder.addBinding(MESSAGE_IDENTIFIER_NAME, muleEventMetadataType.getMessageType());
    return builder;
  }

  public Optional<ModuleDefinition> lookupModule(String moduleName) {
    return modules.stream().filter((module) -> module.getName().toString().equals(moduleName)).findFirst();
  }

  public static class Builder {

    private Map<String, MetadataType> bindings;
    private List<ModuleDefinition.Builder> modules;
    private List<TypeBindings> parents;

    public Builder() {
      bindings = new HashMap<>();
      modules = new ArrayList<>();
      parents = new ArrayList<>();
    }

    public Builder addAll(TypeBindings bindings) {
      parents.add(bindings);
      return this;
    }

    public Builder addBinding(String identifier, MetadataType metadataType) {
      bindings.put(identifier, metadataType);
      return this;
    }

    public ModuleDefinition.Builder module(String name) {
      ModuleDefinition.Builder builder = ModuleDefinition.builder(name);
      modules.add(builder);
      return builder;
    }

    public TypeBindings build() {
      List<ModuleDefinition> resultModules = modules.stream().map(ModuleDefinition.Builder::build).collect(Collectors.toList());
      HashMap<String, MetadataType> resultBindings = new HashMap<>(bindings);
      for (TypeBindings parent : parents) {
        resultBindings.putAll(parent.bindings);
        resultModules.addAll(parent.modules);
      }
      return new TypeBindings(resultBindings, resultModules);
    }

  }
}
