/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.message.api;


import org.mule.metadata.api.annotation.TypeAnnotation;
import org.mule.metadata.api.builder.ObjectFieldTypeBuilder;
import org.mule.metadata.api.model.MetadataFormat;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.api.model.ObjectFieldType;
import org.mule.metadata.api.model.ObjectType;
import org.mule.metadata.api.model.impl.BaseMetadataType;
import org.mule.metadata.api.visitor.MetadataTypeVisitor;

import java.util.Collection;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

import static java.util.stream.Stream.of;

/**
 * Default implementation for {@link MessageMetadataType}.
 *
 * @since 1.0
 */
class DefaultMessageMetadataType extends BaseMetadataType implements ObjectType, MessageMetadataType {

  private static final String PAYLOAD_FIELD_NAME = "payload";
  private static final String ATTRIBUTES_FIELD_NAME = "attributes";
  private final Optional<ObjectFieldType> payloadMetadataType;
  private final Optional<ObjectFieldType> attributes;


  DefaultMessageMetadataType(Optional<MetadataType> payloadMetadataType, Optional<MetadataType> attributes,
                             Map<Class<? extends TypeAnnotation>, TypeAnnotation> annotations) {
    super(MetadataFormat.JAVA, annotations);
    this.payloadMetadataType = payloadMetadataType.map((type) -> {
      final ObjectFieldTypeBuilder messageTypeBuilder = new ObjectFieldTypeBuilder(MetadataFormat.JAVA);
      return messageTypeBuilder.key(PAYLOAD_FIELD_NAME).required(true).value(type).build();
    });

    this.attributes = attributes.map((type) -> {
      final ObjectFieldTypeBuilder variablesTypeBuilder = new ObjectFieldTypeBuilder(MetadataFormat.JAVA);
      variablesTypeBuilder.key(ATTRIBUTES_FIELD_NAME).required(true).value(type);
      return variablesTypeBuilder.build();
    });
  }

  @Override
  public Optional<MetadataType> getOpenRestriction() {
    return Optional.empty();
  }

  @Override
  public Collection<ObjectFieldType> getFields() {
    return of(payloadMetadataType, attributes)
        .filter((t) -> t.isPresent())
        .map((t) -> t.get())
        .collect(Collectors.toList());
  }

  @Override
  public boolean isOrdered() {
    return false;
  }

  @Override
  public Optional<ObjectFieldType> getFieldByName(String name) {
    switch (name) {
      case PAYLOAD_FIELD_NAME:
        return payloadMetadataType;
      case ATTRIBUTES_FIELD_NAME:
        return attributes;
    }
    return Optional.empty();
  }

  @Override
  public void accept(MetadataTypeVisitor metadataTypeVisitor) {
    metadataTypeVisitor.visitObject(this);
  }

  @Override
  public Optional<MetadataType> getPayloadType() {
    return payloadMetadataType.map((t) -> t.getValue());
  }

  @Override
  public Optional<MetadataType> getAttributesType() {
    return attributes.map((t) -> t.getValue());
  }
}
