/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.http.api.domain.entity;

import static org.mule.runtime.api.util.MultiMap.emptyMultiMap;

import static java.util.Collections.emptyList;
import static java.util.OptionalLong.empty;
import static java.util.OptionalLong.of;

import org.mule.runtime.api.util.MultiMap;
import org.mule.runtime.http.api.domain.entity.multipart.HttpPart;

import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.util.Collection;
import java.util.Optional;
import java.util.OptionalLong;
import java.util.function.BiConsumer;
import java.util.function.Consumer;

/**
 * An entity that can be sent or received via an {@link org.mule.runtime.http.api.domain.message.request.HttpRequest} or
 * {@link org.mule.runtime.http.api.domain.message.response.HttpResponse}.
 ** <p>
 * There are three distinct types of entities, depending on their content:
 * <ul>
 * <li><b>streamed</b>: The content is received from a stream.</li>
 * <li><b>composed</b>: The content is made of several parts.</li>
 * <li><b>neither of the above</b>: The content is in memory and whole.</li>
 * </ul>
 * These will determine how the content can be accessed although there may overlap.
 *
 * @since 4.0
 */
public interface HttpEntity {

  /**
   * Tells whether this entity's content is stream based. Streamed entities can only provide their content once, regardless of the
   * access method.
   *
   * @return {@code true} if content is streamed, {@code false} otherwise
   * @see #getContent()
   */
  boolean isStreaming();

  /**
   * Tells whether or not this entity is composed of several parts, in which case they should be available through
   * {@link #getParts()}.
   *
   * @return {@code true} if there are several content parts, {@code false} otherwise
   * @deprecated Use <b>DataWeave</b> to generate multipart content.
   */
  @Deprecated(since = "4.10", forRemoval = true)
  default boolean isComposed() {
    return false;
  }

  /**
   * Provides the entity's content as a stream. All streaming entities should provide their stream.
   *
   * @return an {@code InputStream} representing this entity's content or {@code null} if such representation is not possible
   */
  InputStream getContent();

  /**
   * Provides the entity's content as bytes. If the entity is stream based, then the stream will be consumed as a consequence.
   *
   * @return a byte array representing this entity's content or {@code null} if such representation is not possible
   * @throws IOException if an error occurs creating the byte array
   */
  byte[] getBytes() throws IOException;

  /**
   * Provides the entity's content parts. If the entity is stream based, then the stream will be consumed as a consequence. Non
   * composed entities should return an empty collection.
   *
   * @return a collection of {@link HttpPart HttpParts} representing this entity's content parts, if present
   * @throws IOException if an error occurs handling the parts
   * @see #isComposed()
   * @deprecated Use <b>DataWeave</b> to generate multipart content.
   */
  @Deprecated(since = "4.10", forRemoval = true)
  default Collection<HttpPart> getParts() throws IOException {
    return emptyList();
  }

  /**
   * Provides the length (in bytes) of the {@link HttpEntity}, if known. For the most part, only received entities from HTTP
   * messages that carried a 'Content-Length' header will return a length.
   *
   * @return an {@link Optional} with the length (in bytes) or an empty one if unknown
   *
   * @deprecated Use {@link #getBytesLength()} instead.
   */
  @Deprecated
  Optional<Long> getLength();

  /**
   * Provides the length (in bytes) of the {@link HttpEntity}, if known. For the most part, only received entities from HTTP
   * messages that carried a 'Content-Length' header will return a length.
   *
   * @return an {@link Optional} with the length (in bytes) or an empty one if unknown
   *
   * @since 4.2
   */
  default OptionalLong getBytesLength() {
    return getLength().map(l -> of(l)).orElse(empty());
  }

  /**
   * To be called when the trailers are received, or when we know that we aren't going to receive any trailers. In that case, the
   * parameter of the consumer will be an empty {@link MultiMap}. For example, for HTTP/1 messages, we know that we will never
   * receive trailers, so this callback will be preemptively invoked.
   *
   * @param completionCallback callback to be invoked.
   *
   * @since 4.11
   */
  default void onComplete(BiConsumer<? super MultiMap<String, String>, ? super Throwable> completionCallback) {
    completionCallback.accept(emptyMultiMap(), null);
  }

  /**
   * Handle each fragment of data.
   *
   * @param dataConsumer callback to be executed for each data fragment seen in the entity.
   * @return a {@link EntitySubscription} object that can be used to cancel the reactive consumption of the entity.
   * @throws UnsupportedOperationException if the implementation doesn't support reactive consumption.
   *
   * @since 4.11
   */
  default EntitySubscription onData(Consumer<ByteBuffer> dataConsumer) {
    throw new UnsupportedOperationException("This entity does not support reactive data fragments consuming");
  }

  /**
   * @return whether this entity supports consuming data fragments reactively, by using {@link #onData(Consumer)}
   * @since 4.11
   * @see #onData(Consumer)
   */
  default boolean isReactive() {
    return false;
  }
}
