/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.http.api.domain.entity;

import static org.mule.runtime.api.util.MultiMap.emptyMultiMap;

import static java.util.Optional.of;
import static java.util.concurrent.CompletableFuture.completedFuture;

import org.mule.runtime.api.util.MultiMap;

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.util.Optional;
import java.util.OptionalLong;
import java.util.concurrent.CompletableFuture;
import java.util.function.BiConsumer;

/**
 * Represents an HTTP empty body.
 *
 * @since 4.0
 */
public final class EmptyHttpEntity implements HttpEntity {

  private static final byte[] NO_BYTES = new byte[0];

  private final CompletableFuture<MultiMap<String, String>> futureTrailers;

  public EmptyHttpEntity() {
    this(completedFuture(emptyMultiMap()));
  }

  public EmptyHttpEntity(CompletableFuture<MultiMap<String, String>> futureTrailers) {
    this.futureTrailers = futureTrailers;
  }

  @Override
  public boolean isStreaming() {
    return false;
  }

  @Override
  public InputStream getContent() {
    return new ByteArrayInputStream(NO_BYTES);
  }

  @Override
  public byte[] getBytes() {
    return NO_BYTES;
  }

  @Override
  public Optional<Long> getLength() {
    return of(0L);
  }

  @Override
  public OptionalLong getBytesLength() {
    return OptionalLong.of(0L);
  }

  @Override
  public void onComplete(BiConsumer<? super MultiMap<String, String>, ? super Throwable> completionCallback) {
    futureTrailers.whenComplete(completionCallback);
  }
}
