/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.features.internal.generator;

import static org.assertj.core.api.Assertions.assertThat;

import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.io.TempDir;

import java.io.File;
import java.io.IOException;
import java.net.URISyntaxException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.List;

/**
 * Test class for {@link MuleFeaturesGenerator} that validates the generated output against expected structure and content.
 */
public class MuleFeaturesGeneratorTest {

  @TempDir
  Path tempDir;

  private MuleFeaturesGenerator generator;
  private File outputDir;

  @BeforeEach
  void setUp() {
    outputDir = tempDir.toFile();
    generator = new MuleFeaturesGenerator(outputDir);
  }

  @Test
  void testGeneratedFileContainsExpectedResourceFeatures() throws IOException, URISyntaxException {
    generator.generate();

    Path generatedFile = tempDir.resolve("org/mule/runtime/features/api/MuleRuntimeFeature.java");
    Path expectedResourceFile = Paths.get(getClass().getResource("/TestingMuleRuntimeFeature.java").toURI());

    assertThat(generatedFile).exists();

    String generatedContent = Files.readString(generatedFile);
    String expectedContent = Files.readString(expectedResourceFile);

    List<String> expectedFeatureNames = extractFeatureNames(expectedContent);
    for (String featureName : expectedFeatureNames) {
      assertThat(generatedContent).as("Generated file should contain feature: " + featureName)
          .contains(featureName);
    }

    String[] requiredMethods = {
        "public String getDescription()",
        "public String getIssueId()",
        "public String getSince()",
        "public String getEnabledByDefaultSince()",
        "public Optional<String> getOverridingSystemPropertyName()",
        "public Optional<String> getMinJavaVersion()"
    };

    for (String method : requiredMethods) {
      assertThat(generatedContent).as("Generated file should contain method: " + method).contains(method);
      assertThat(expectedContent).as("Expected resource file should contain method: " + method).contains(method);
    }

    if (expectedContent.contains("MULE-17659")) {
      assertThat(generatedContent).containsPattern(".*MULE-17659.*");
    }
    if (expectedContent.contains("DW-383")) {
      assertThat(generatedContent).containsPattern(".*DW-383.*");
    }
  }

  /**
   * Helper method to extract feature names from file content. Looks for enum entries like "FEATURE_NAME(" and extracts the
   * feature name.
   */
  private List<String> extractFeatureNames(String content) {
    List<String> featureNames = new java.util.ArrayList<>();

    String[] lines = content.split("\n");
    for (String line : lines) {
      String trimmedLine = line.trim();
      if (trimmedLine.matches("^[A-Z_]+\\(.*")) {
        String featureName = trimmedLine.substring(0, trimmedLine.indexOf('('));
        featureNames.add(featureName);
      }
    }

    return featureNames;
  }
}
