/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.extension.api.test.internal.loader.util;

import static org.hamcrest.CoreMatchers.is;
import static org.hamcrest.CoreMatchers.notNullValue;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.mule.runtime.extension.api.ExtensionConstants.TLS_PARAMETER_NAME;
import static org.mule.runtime.extension.internal.loader.util.InfrastructureTypeMapping.TLS_KEY_STORE_PARAMETER_NAME;
import static org.mule.runtime.extension.internal.loader.util.InfrastructureTypeMapping.TLS_TRUST_STORE_PARAMETER_NAME;
import static org.mule.runtime.internal.dsl.DslConstants.TLS_KEY_STORE_ELEMENT_IDENTIFIER;
import static org.mule.runtime.internal.dsl.DslConstants.TLS_TRUST_STORE_ELEMENT_IDENTIFIER;
import static org.mule.runtime.extension.api.declaration.type.StreamingStrategyTypeBuilder.REPEATABLE_FILE_STORE_BYTES_STREAM_ALIAS;
import static org.mule.runtime.extension.api.declaration.type.StreamingStrategyTypeBuilder.REPEATABLE_IN_MEMORY_BYTES_STREAM_ALIAS;
import static org.mule.runtime.extension.api.declaration.type.StreamingStrategyTypeBuilder.NON_REPEATABLE_BYTE_STREAM_ALIAS;

import org.mule.runtime.api.meta.model.ParameterDslConfiguration;
import org.mule.runtime.api.tls.TlsContextFactory;
import org.mule.runtime.api.tls.TlsContextKeyStoreConfiguration;
import org.mule.runtime.api.tls.TlsContextTrustStoreConfiguration;
import org.mule.runtime.extension.api.property.QNameModelProperty;
import org.mule.runtime.extension.internal.loader.util.InfrastructureTypeMapping;
import org.mule.runtime.extension.internal.loader.util.InfrastructureTypeMapping.InfrastructureType;

import java.util.Optional;

import org.junit.Test;

/**
 * Unit tests for {@link InfrastructureTypeMapping}
 *
 * @since 4.0
 */
public class InfrastructureTypeMappingTestCase {

  @Test
  public void tlsContextFactoryIsMapped() {
    // Test that TlsContextFactory is properly mapped
    InfrastructureType tlsInfraType = InfrastructureTypeMapping.getMap().get(TlsContextFactory.class);
    assertThat(tlsInfraType, is(notNullValue()));
    assertThat(tlsInfraType.getName(), is(TLS_PARAMETER_NAME));
    assertThat(tlsInfraType.getSequence(), is(8));
  }

  @Test
  public void tlsKeyStoreConfigurationIsMapped() {
    // Test that TlsContextKeyStoreConfiguration is properly mapped
    InfrastructureType keyStoreInfraType = InfrastructureTypeMapping.getMap().get(TlsContextKeyStoreConfiguration.class);
    assertThat(keyStoreInfraType, is(notNullValue()));
    assertThat(keyStoreInfraType.getName(), is(TLS_KEY_STORE_PARAMETER_NAME));
    assertThat(keyStoreInfraType.getSequence(), is(12));
  }

  @Test
  public void tlsTrustStoreConfigurationIsMapped() {
    // Test that TlsContextTrustStoreConfiguration is properly mapped
    InfrastructureType trustStoreInfraType = InfrastructureTypeMapping.getMap().get(TlsContextTrustStoreConfiguration.class);
    assertThat(trustStoreInfraType, is(notNullValue()));
    assertThat(trustStoreInfraType.getName(), is(TLS_TRUST_STORE_PARAMETER_NAME));
    assertThat(trustStoreInfraType.getSequence(), is(13));
  }

  @Test
  public void tlsKeyStoreQNameIsConfigured() {
    // Test that KeyStore has proper QName configuration
    Optional<QNameModelProperty> qName = InfrastructureTypeMapping.getQName(TLS_KEY_STORE_PARAMETER_NAME);
    assertThat(qName.isPresent(), is(true));
    assertThat(qName.get().getValue().getLocalPart(), is(TLS_KEY_STORE_ELEMENT_IDENTIFIER));
    assertThat(qName.get().getValue().getNamespaceURI(), is(InfrastructureTypeMapping.TLS_NAMESPACE_URI));
  }

  @Test
  public void tlsTrustStoreQNameIsConfigured() {
    // Test that TrustStore has proper QName configuration
    Optional<QNameModelProperty> qName = InfrastructureTypeMapping.getQName(TLS_TRUST_STORE_PARAMETER_NAME);
    assertThat(qName.isPresent(), is(true));
    assertThat(qName.get().getValue().getLocalPart(), is(TLS_TRUST_STORE_ELEMENT_IDENTIFIER));
    assertThat(qName.get().getValue().getNamespaceURI(), is(InfrastructureTypeMapping.TLS_NAMESPACE_URI));
  }

  @Test
  public void tlsKeyStoreDslConfigurationIsConfigured() {
    // Test that KeyStore has proper DSL configuration
    Optional<ParameterDslConfiguration> dslConfig = InfrastructureTypeMapping.getDslConfiguration(TLS_KEY_STORE_PARAMETER_NAME);
    assertThat(dslConfig.isPresent(), is(true));
    assertThat(dslConfig.get().allowsInlineDefinition(), is(true));
    assertThat(dslConfig.get().allowTopLevelDefinition(), is(false));
    assertThat(dslConfig.get().allowsReferences(), is(false));
  }

  @Test
  public void tlsTrustStoreDslConfigurationIsConfigured() {
    // Test that TrustStore has proper DSL configuration
    Optional<ParameterDslConfiguration> dslConfig = InfrastructureTypeMapping.getDslConfiguration(TLS_TRUST_STORE_PARAMETER_NAME);
    assertThat(dslConfig.isPresent(), is(true));
    assertThat(dslConfig.get().allowsInlineDefinition(), is(true));
    assertThat(dslConfig.get().allowTopLevelDefinition(), is(false));
    assertThat(dslConfig.get().allowsReferences(), is(false));
  }

  @Test
  public void streamingStrategyMappingsAreConfigured() {
    // Test that streaming strategy QNames are properly configured
    Optional<QNameModelProperty> fileStoreQName = InfrastructureTypeMapping.getQName(REPEATABLE_FILE_STORE_BYTES_STREAM_ALIAS);
    assertThat(fileStoreQName.isPresent(), is(true));
    assertThat(fileStoreQName.get().getValue().getLocalPart(), is("repeatable-file-store-stream"));

    Optional<QNameModelProperty> inMemoryQName = InfrastructureTypeMapping.getQName(REPEATABLE_IN_MEMORY_BYTES_STREAM_ALIAS);
    assertThat(inMemoryQName.isPresent(), is(true));
    assertThat(inMemoryQName.get().getValue().getLocalPart(), is("repeatable-in-memory-stream"));

    Optional<QNameModelProperty> nonRepeatableQName = InfrastructureTypeMapping.getQName(NON_REPEATABLE_BYTE_STREAM_ALIAS);
    assertThat(nonRepeatableQName.isPresent(), is(true));
    assertThat(nonRepeatableQName.get().getValue().getLocalPart(), is("non-repeatable-stream"));
  }
}
