/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.extension.internal;

import static org.mule.runtime.ast.extension.internal.ExtensionTypeUtils.isLiteralType;
import static org.mule.runtime.ast.extension.internal.ExtensionTypeUtils.isParameterResolverType;
import static org.mule.runtime.ast.extension.internal.ExtensionTypeUtils.isTypedValueType;

import org.mule.metadata.api.builder.TypeBuilder;
import org.mule.metadata.ast.api.IntrospectionContext;
import org.mule.metadata.ast.api.TypeHandler;

import javax.annotation.processing.ProcessingEnvironment;
import javax.lang.model.type.DeclaredType;
import javax.lang.model.type.TypeMirror;
import javax.lang.model.type.TypeVisitor;

import java.util.List;

/**
 * {@link TypeHandler} implementation for Extension Types.
 *
 * @since 1.0
 */
public class ExtensionTypeHandler implements TypeHandler {

  private final ProcessingEnvironment processingEnvironment;

  public ExtensionTypeHandler(ProcessingEnvironment processingEnvironment) {
    this.processingEnvironment = processingEnvironment;
  }

  @Override
  public boolean handles(TypeMirror typeMirror) {
    return typeMirror instanceof DeclaredType
        && isHandled(typeMirror);
  }

  private boolean isHandled(TypeMirror typeMirror) {
    return isParameterResolverType(typeMirror, processingEnvironment) ||
        isLiteralType(typeMirror, processingEnvironment) ||
        isTypedValueType(typeMirror, processingEnvironment);
  }

  @Override
  public TypeBuilder<?> handle(TypeMirror typeMirror, TypeVisitor<TypeBuilder<?>, IntrospectionContext> typeVisitor,
                               IntrospectionContext context) {
    if (typeMirror instanceof DeclaredType declaredType && isHandled(declaredType)) {
      List<? extends TypeMirror> typeArguments = declaredType.getTypeArguments();
      if (typeArguments.isEmpty()) {
        throw new IllegalArgumentException("The given TypeMirror " + declaredType + " doesn't provide generics");
      }
      return typeArguments.get(0).accept(typeVisitor, context);
    }
    throw new IllegalArgumentException("The given TypeMirror " + typeMirror + " it's not supported by this Handler");
  }


}
