/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal;

import static java.lang.String.format;
import static java.util.Collections.emptyList;

import java.util.List;

import javax.annotation.processing.ProcessingEnvironment;
import javax.lang.model.element.TypeElement;
import javax.lang.model.type.DeclaredType;
import javax.lang.model.type.TypeMirror;
import javax.lang.model.util.Types;

/**
 * Type Utilities
 *
 * @since 1.1
 */
public class IntrospectionTypeUtils {

  private IntrospectionTypeUtils() {}

  public static List<? extends TypeMirror> getSuperTypeGenerics(TypeMirror type, Class superClass,
                                                                ProcessingEnvironment processingEnvironment) {

    TypeElement superClassTypeElement = processingEnvironment.getElementUtils().getTypeElement(superClass.getCanonicalName());
    TypeElement objectType = processingEnvironment.getElementUtils().getTypeElement(Object.class.getName());
    TypeMirror superClassTypeMirror = processingEnvironment.getTypeUtils().erasure(superClassTypeElement.asType());

    if (!processingEnvironment.getTypeUtils().isAssignable(type, superClassTypeMirror)) {
      throw new IllegalArgumentException(
                                         format("Class '%s' does not extend the '%s' class", type.toString(),
                                                superClass.getSimpleName()));
    }

    if (isSameType(type, processingEnvironment, superClassTypeMirror)) {
      return getGenericTypes(type);
    }

    DeclaredType searchClass = (DeclaredType) type;
    while (!processingEnvironment.getTypeUtils().isAssignable(objectType.asType(), searchClass)) {
      for (TypeMirror typeMirror : processingEnvironment.getTypeUtils().directSupertypes(searchClass)) {
        if (isSameType(typeMirror, processingEnvironment, superClassTypeMirror)) {
          return getGenericTypes(typeMirror);
        } else {
          if (typeMirror instanceof DeclaredType
              && processingEnvironment.getTypeUtils().isAssignable(typeMirror, superClassTypeMirror)) {
            searchClass = (DeclaredType) typeMirror;
          }
        }
      }
    }
    return emptyList();
  }

  private static List<? extends TypeMirror> getGenericTypes(TypeMirror typeMirror) {
    if (typeMirror instanceof DeclaredType) {
      return ((DeclaredType) typeMirror).getTypeArguments();
    } else {
      return emptyList();
    }
  }

  private static boolean isSameType(TypeMirror type, ProcessingEnvironment processingEnvironment,
                                    TypeMirror superClassTypeMirror) {
    return processingEnvironment.getTypeUtils().isSameType(processingEnvironment.getTypeUtils().erasure(type),
                                                           superClassTypeMirror);
  }

  public static List<? extends TypeMirror> getSuperTypeGenerics(TypeElement type, Class superClass,
                                                                ProcessingEnvironment processingEnvironment) {

    return getSuperTypeGenerics(type.asType(), superClass, processingEnvironment);
  }

  /**
   * Tests whether {@code typeElement} represents a subtype of {@code baseType}. This is plain old class inheritance, generics
   * compatibility is not considered.
   *
   * @param typeElement           the potential subtype
   * @param baseType              the base type
   * @param processingEnvironment the current processing environment
   * @return {@code} true if a subtype
   * @since 1.3.0
   */
  public static boolean isSubType(TypeElement typeElement, Class<?> baseType, ProcessingEnvironment processingEnvironment) {
    return isErasureSubType(typeElement,
                            processingEnvironment.getElementUtils().getTypeElement(baseType.getCanonicalName()),
                            processingEnvironment);
  }

  /**
   * Tests whether {@code typeElement} represents a subtype of {@code baseType}. This is plain old class inheritance, generics
   * compatibility is not considered.
   *
   * @param typeElement           the potential subtype
   * @param baseType              the base type
   * @param processingEnvironment the current processing environment
   * @return {@code} true if a subtype
   * @since 1.3.0
   */
  public static boolean isErasureSubType(TypeElement typeElement,
                                         TypeElement baseType,
                                         ProcessingEnvironment processingEnvironment) {
    final Types typeUtils = processingEnvironment.getTypeUtils();
    final TypeMirror baseMirror = typeUtils.erasure(baseType.asType());

    return typeUtils.isSubtype(typeUtils.erasure(typeElement.asType()), baseMirror);
  }
}
