/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal;

import static javax.lang.model.element.ElementKind.INTERFACE;

import org.mule.metadata.api.annotation.TypeAliasAnnotation;
import org.mule.metadata.api.builder.ObjectFieldTypeBuilder;
import org.mule.metadata.api.builder.ObjectTypeBuilder;
import org.mule.metadata.api.builder.TypeBuilder;
import org.mule.metadata.ast.api.IntrospectionContext;
import org.mule.metadata.ast.api.ObjectFieldHandler;
import org.mule.metadata.ast.internal.DefaultObjectFieldHandler;
import org.mule.runtime.api.util.collection.SmallMap;
import org.mule.runtime.extension.api.annotation.Alias;
import org.mule.runtime.extension.api.annotation.Extensible;
import org.mule.runtime.extension.api.annotation.dsl.xml.TypeDsl;
import org.mule.runtime.extension.api.declaration.type.annotation.ExtensibleTypeAnnotation;
import org.mule.runtime.extension.api.declaration.type.annotation.TypeDslAnnotation;
import org.mule.runtime.extension.api.runtime.route.Chain;
import org.mule.runtime.module.extension.api.loader.java.type.ExtensionParameter;
import org.mule.runtime.module.extension.api.loader.java.type.ParameterizableTypeElement;
import org.mule.runtime.module.extension.api.loader.java.type.Type;

import java.util.List;
import java.util.Map;
import java.util.Optional;

import javax.annotation.processing.ProcessingEnvironment;
import javax.lang.model.element.AnnotationMirror;
import javax.lang.model.element.Element;
import javax.lang.model.element.TypeElement;
import javax.lang.model.type.TypeVisitor;

/**
 * {@link ObjectFieldHandler} implementation for Extension Types.
 * <p>
 * This is a reduced and AST based version of the ExtensionsObjectFieldHandler of the Mule Extensions API.
 *
 * @since 1.0
 */
public final class ExtensionTypeObjectFieldHandler implements ObjectFieldHandler {

  private ProcessingEnvironment processingEnvironment;

  ExtensionTypeObjectFieldHandler(ProcessingEnvironment processingEnvironment) {
    this.processingEnvironment = processingEnvironment;
  }

  @Override
  public void handle(Element element, ObjectTypeBuilder builder, IntrospectionContext context,
                     TypeVisitor<TypeBuilder<?>, IntrospectionContext> typeVisitor) {
    if (element instanceof TypeElement) {
      ParameterizableTypeElement type = new ASTParameterizableType((TypeElement) element, processingEnvironment);
      if (type.isSameType(Chain.class)) {
        return;
      }

      processAnnotations(element, builder);

      List<ExtensionParameter> parameters = type.getParameters();
      if (parameters.isEmpty()) {
        fallbackToBeanProperties(element, builder, context, typeVisitor);
        return;
      }

      for (ExtensionParameter parameter : parameters) {
        final ObjectFieldTypeBuilder fieldBuilder = builder.addField();
        fieldBuilder.key(parameter.getAlias());
        parameter.getElement().ifPresent(e -> extractJavadoc(processingEnvironment, e)
            .ifPresent(fieldBuilder::description));
        setFieldType(parameter, fieldBuilder, context, typeVisitor);
      }
    }
  }

  private void fallbackToBeanProperties(Element element, ObjectTypeBuilder builder, IntrospectionContext context,
                                        TypeVisitor<TypeBuilder<?>, IntrospectionContext> typeVisitor) {
    if (!element.getKind().equals(INTERFACE)) {
      new DefaultObjectFieldHandler(processingEnvironment).handle(element, builder, context, typeVisitor);
    }
  }

  private void setFieldType(ExtensionParameter parameter, ObjectFieldTypeBuilder fieldBuilder, IntrospectionContext context,
                            TypeVisitor<TypeBuilder<?>, IntrospectionContext> typeVisitor) {
    Type type = parameter.getType();
    Optional<TypeElement> optionalElement = type.getElement();
    if (optionalElement.isPresent()) {
      TypeElement typeElement = optionalElement.get();
      if (context.contains(typeElement)) {
        fieldBuilder.value(context.get(typeElement));
      } else {
        fieldBuilder.value(((ASTType) type).getTypeMirror().accept(typeVisitor, context));
      }
    } else {
      fieldBuilder.value(type.asMetadataType());
    }
  }

  private Optional<String> extractJavadoc(ProcessingEnvironment processingEnv, Element element) {
    String comment = processingEnv.getElementUtils().getDocComment(element);
    if (comment == null || comment.trim().isEmpty()) {
      return Optional.empty();
    }

    return Optional.of(comment.trim().replaceAll("\\{@.+ (.+)\\}", "$1"));
  }

  private void processAnnotations(Element element, ObjectTypeBuilder builder) {
    processExtensible(element, builder);
    processTypeAlias(element, builder);
    processTypeDsl(element, builder);
  }

  private void processExtensible(Element element, ObjectTypeBuilder builder) {
    if (getAnnotation(element, Extensible.class.getName()).isPresent() ||
        getAnnotation(element, org.mule.sdk.api.annotation.Extensible.class.getName()).isPresent()) {
      builder.with(new ExtensibleTypeAnnotation());
    }
  }

  private void processTypeAlias(Element element, ObjectTypeBuilder builder) {
    Map<String, Object> annotationValues = getAnnotationValues(element,
                                                               Alias.class.getName(),
                                                               org.mule.sdk.api.annotation.Alias.class.getName());

    if (!annotationValues.isEmpty()) {
      builder.with(new TypeAliasAnnotation((String) annotationValues.get("value")));
    }
  }

  private void processTypeDsl(Element element, ObjectTypeBuilder builder) {
    Map<String, Object> annotationValues =
        getAnnotationValues(element, TypeDsl.class.getName());
    if (!annotationValues.isEmpty()) {
      builder.with(new TypeDslAnnotation((Boolean) annotationValues.getOrDefault("allowInlineDefinition", true),
                                         (Boolean) annotationValues.getOrDefault("allowTopLevelDefinition", false),
                                         (String) annotationValues.getOrDefault("substitutionGroup", ""),
                                         (String) annotationValues.getOrDefault("baseType", "")));
    }
  }

  private Optional<? extends AnnotationMirror> getAnnotation(Element element, String annotationName) {
    return element.getAnnotationMirrors().stream().filter(a -> a.getAnnotationType().toString().equals(annotationName))
        .findFirst();
  }

  private Map<String, Object> getAnnotationValues(Element element, String... annotationNames) {
    Map<String, Object> annotationValues = new SmallMap<>();

    for (String annotationName : annotationNames) {
      Optional<? extends AnnotationMirror> annotationMirror = getAnnotation(element, annotationName);
      if (annotationMirror.isPresent()) {
        AnnotationMirror a = annotationMirror.get();
        a.getElementValues().forEach((key, value) -> annotationValues.put(key.getSimpleName().toString(), value.getValue()));
        break;
      }
    }

    return annotationValues;
  }
}
