/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.embedded.commons.internal;

import org.mule.runtime.module.embedded.commons.api.DeploymentConfiguration;
import org.mule.runtime.module.embedded.commons.api.DeploymentConfiguration.DeploymentConfigurationBuilder;

/**
 * Default implementation of {@link DeploymentConfigurationBuilder}.
 */
public class DefaultDeploymentConfigurationBuilder implements DeploymentConfigurationBuilder {

  private boolean lazyInitializationEnabled = false;
  private boolean xmlValidationsEnabled = false;
  private boolean lazyConnectionsEnabled = false;
  private boolean toolingObjectsNotToRegistry = false;
  private boolean addArtifactAstToRegistry = false;

  @Override
  public DeploymentConfigurationBuilder lazyInitialization(boolean enabled) {
    lazyInitializationEnabled = enabled;
    return this;
  }

  @Override
  public DeploymentConfigurationBuilder xmlValidations(boolean enabled) {
    xmlValidationsEnabled = enabled;
    return this;
  }

  @Override
  public DeploymentConfigurationBuilder lazyConnectionsEnabled(boolean enabled) {
    lazyConnectionsEnabled = enabled;
    return this;
  }

  @Override
  public DeploymentConfigurationBuilder doNotAddToolingObjectsToRegistry(boolean toolingObjectsNotToRegistry) {
    this.toolingObjectsNotToRegistry = toolingObjectsNotToRegistry;
    return this;
  }

  @Override
  public DeploymentConfigurationBuilder addArtifactAstToRegistry(boolean addArtifactAstToRegistry) {
    this.addArtifactAstToRegistry = addArtifactAstToRegistry;
    return this;
  }

  @Override
  public DeploymentConfiguration build() {
    return new DeploymentConfigurationImpl(lazyInitializationEnabled, xmlValidationsEnabled, lazyConnectionsEnabled,
                                           toolingObjectsNotToRegistry, addArtifactAstToRegistry);
  }

  static class DeploymentConfigurationImpl implements DeploymentConfiguration {

    private static final long serialVersionUID = 3520767160861854299L;

    private final boolean lazyInitializationEnabled;
    private final boolean xmlValidationsEnabled;
    private final boolean lazyConnectionsEnabled;
    private final boolean toolingObjectsNotToRegistry;
    private final boolean addArtifactAstToRegistry;

    public DeploymentConfigurationImpl(boolean lazyInitializationEnabled, boolean xmlValidationsEnabled,
                                       boolean lazyConnectionsEnabled,
                                       boolean toolingObjectsNotToRegistry,
                                       boolean addArtifactAstToRegistry) {
      this.lazyInitializationEnabled = lazyInitializationEnabled;
      this.xmlValidationsEnabled = xmlValidationsEnabled;
      this.lazyConnectionsEnabled = lazyConnectionsEnabled;
      this.toolingObjectsNotToRegistry = toolingObjectsNotToRegistry;
      this.addArtifactAstToRegistry = addArtifactAstToRegistry;
    }

    @Override
    public boolean lazyInitializationEnabled() {
      return lazyInitializationEnabled;
    }

    @Override
    public boolean xmlValidationsEnabled() {
      return xmlValidationsEnabled;
    }

    @Override
    public boolean lazyConnectionsEnabled() {
      return lazyConnectionsEnabled;
    }

    @Override
    public boolean doNotAddToolingObjectsToRegistry() {
      return toolingObjectsNotToRegistry;
    }

    @Override
    public boolean addArtifactAstToRegistry() {
      return addArtifactAstToRegistry;
    }
  }
}
