/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.embedded.internal;

import static org.mule.runtime.module.embedded.api.ContainerInformation.fromContainerClassLoader;
import static org.mule.runtime.module.embedded.internal.utils.EmbeddedImplementationUtils.createEmbeddedImplClassLoader;

import static java.lang.Thread.currentThread;
import static java.util.concurrent.Executors.newSingleThreadExecutor;

import static org.slf4j.LoggerFactory.getLogger;

import org.mule.maven.client.api.MavenClient;
import org.mule.runtime.module.embedded.api.ContainerConfiguration;
import org.mule.runtime.module.embedded.api.ContainerInfo;
import org.mule.runtime.module.embedded.api.ContainerInformation;
import org.mule.runtime.module.embedded.api.DeploymentService;
import org.mule.runtime.module.embedded.api.EmbeddedContainer;
import org.mule.runtime.module.embedded.internal.dependencies.DefaultDependencyResolver;

import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.net.URL;
import java.util.List;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;

import org.slf4j.Logger;

/**
 * Common functionality for implementations of {@link EmbeddedContainer}.
 *
 * @param <T> the type of embedded controller used by the implementation.
 */
public abstract class AbstractEmbeddedContainer<T> implements EmbeddedContainer {

  private static final String BOOT_LICENSE_PACKAGE = "com.mulesoft.mule.runtime.module.boot.license";

  protected static final Logger LOGGER = getLogger(AbstractEmbeddedContainer.class);

  private final String muleVersion;
  private final ContainerConfiguration containerConfiguration;
  private final List<URL> serverPlugins;
  private final URL containerBaseFolder;
  private final MavenClient mavenClient;
  private final List<URL> services;
  private final ContainerInformation containerInformation;

  private ClassLoader containerModulesClassLoader;
  private boolean started = false;
  private T embeddedController;
  private ClassLoader embeddedControllerBootstrapClassLoader;

  // Executor service to use for running the container operations. The agent is using jetty which uses thread locals that
  // end up
  // with references to class loaders of the container and that causes a memory leak if the thread is never disposed. This
  // guarantees
  // that every operation happens in the thread of this executor service or a thread created by the runtime.
  private ExecutorService executorService;

  public AbstractEmbeddedContainer(String muleVersion, ContainerConfiguration containerConfiguration,
                                   ClassLoader containerModulesClassLoader,
                                   List<URL> services, List<URL> serverPlugins, URL containerBaseFolder,
                                   MavenClient mavenClient) {
    this.muleVersion = muleVersion;
    this.containerConfiguration = containerConfiguration;
    this.containerModulesClassLoader = containerModulesClassLoader;
    this.services = services;
    this.serverPlugins = serverPlugins;
    this.containerBaseFolder = containerBaseFolder;
    this.mavenClient = mavenClient;
    this.containerInformation = fromContainerClassLoader(containerModulesClassLoader);
  }

  @Override
  public synchronized void start() {
    if (!started) {
      try {
        embeddedControllerBootstrapClassLoader =
            createEmbeddedImplClassLoader(containerModulesClassLoader, new DefaultDependencyResolver(mavenClient),
                                          getMuleContainerVersion());

        ContainerInfo containerInfo = new ContainerInfo(muleVersion, containerBaseFolder, services, serverPlugins);

        startExecutorService();

        embeddedController = getEmbeddedController(embeddedControllerBootstrapClassLoader, containerInfo);
      } catch (Exception e) {
        // Clean up resources by calling stop in case of error during start
        try {
          doStop();
        } catch (Exception eStop) {
          e.addSuppressed(eStop);
        }
        throw new IllegalStateException("Cannot start embedded container", e);
      }
      started = true;
      startEmbeddedController(embeddedController);
    } else {
      if (LOGGER.isDebugEnabled()) {
        LOGGER.debug("Embedded container already started");
      }
    }
  }

  @Override
  public synchronized void stop() {
    if (started) {
      try {
        doStop();
      } finally {
        started = false;
      }
    } else {
      if (LOGGER.isDebugEnabled()) {
        LOGGER.debug("Embedded container already stopped");
      }
    }
  }

  private void doStop() {
    if (embeddedController != null) {
      stopEmbeddedController(embeddedController);
      embeddedController = null;
    }

    try {
      mavenClient.close();
    } catch (Exception e) {
      LOGGER.error("Error while closing 'mavenClient'", e);
    }

    stopExecutorService();
    disposeTestingModeInterrupter();

    closeClassLoader(containerModulesClassLoader);
    containerModulesClassLoader = null;
    closeClassLoader(embeddedControllerBootstrapClassLoader);
    embeddedControllerBootstrapClassLoader = null;
  }

  // The product may not be MuleRuntime, but something that contains it. Since the embedded-impl follows the version
  // schema of the runtime, we have to obtain it instead of assuming it is the same as the product's.
  @Override
  public String getMuleContainerVersion() {
    return containerInformation.getMuleContainerVersion();
  }

  @Override
  public DeploymentService getDeploymentService() {
    return doGetDeploymentService(embeddedController);
  }

  @Override
  public File getContainerFolder() {
    return containerConfiguration.getContainerFolder();
  }

  @Override
  public boolean isCurrentJvmVersionSupported() {
    return containerInformation.isCurrentJvmVersionSupported();
  }

  @Override
  public boolean isCurrentJvmVersionRecommended() {
    return containerInformation.isCurrentJvmVersionRecommended();
  }

  protected abstract void startEmbeddedController(T embeddedController);

  protected abstract void stopEmbeddedController(T embeddedController);

  protected abstract DeploymentService doGetDeploymentService(T embeddedController);

  protected abstract T getEmbeddedController(ClassLoader embeddedControllerBootstrapClassLoader, ContainerInfo containerInfo)
      throws Exception;

  private void startExecutorService() {
    executorService = newSingleThreadExecutor(runnable -> new Thread(runnable, "Embedded"));
  }

  private void stopExecutorService() {
    if (executorService != null) {
      executorService.shutdownNow();
      executorService = null;
    }
  }

  private void disposeTestingModeInterrupter() {
    try {
      final Class interrupterClass =
          containerModulesClassLoader.loadClass(BOOT_LICENSE_PACKAGE + ".interrupt.TestingModeInterrupter");
      final Object interrupter = interrupterClass.getDeclaredMethod("getInterrupter").invoke(null);
      interrupterClass.getDeclaredMethod("dispose").invoke(interrupter);
    } catch (ClassNotFoundException | IllegalAccessException | NoSuchMethodException e) {
      // this will happen for versions < 4.10 since it was handled elsewhere back then
      LOGGER.debug("Unable to dispose testingMode interrupter: {}", e.toString());
    } catch (InvocationTargetException e) {
      LOGGER.error("Error while disposing 'TestingModeInterrupter'", e.getCause());
    }
  }

  private void closeClassLoader(ClassLoader classLoader) {
    if (classLoader instanceof Closeable) {
      try {
        ((Closeable) classLoader).close();
      } catch (IOException e) {
        // Do nothing.
        if (LOGGER.isDebugEnabled()) {
          LOGGER.debug("Failure closing container classloader", e);
        }
      }
    }
  }

  protected void executeUsingExecutorService(Callable<Void> callable) {
    Future<?> future = executorService.submit(() -> {
      ClassLoader contextClassLoader = currentThread().getContextClassLoader();
      try {
        currentThread().setContextClassLoader(containerModulesClassLoader);

        callable.call();
      } catch (Exception e) {
        throw new IllegalStateException(e);
      } finally {
        currentThread().setContextClassLoader(contextClassLoader);
      }
    });
    try {
      future.get();
    } catch (Exception e) {
      throw new RuntimeException(e);
    }
  }

}
