/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.embedded.internal.utils;

import static org.mule.maven.pom.parser.api.model.BundleScope.PROVIDED;

import static java.lang.System.getProperty;
import static java.util.Arrays.stream;
import static java.util.Collections.emptyList;
import static java.util.Collections.singletonList;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toSet;

import static org.apache.commons.lang3.JavaVersion.JAVA_17;
import static org.apache.commons.lang3.SystemUtils.isJavaVersionAtLeast;
import static org.slf4j.LoggerFactory.getLogger;

import org.mule.maven.pom.parser.api.model.BundleDependency;
import org.mule.maven.pom.parser.api.model.BundleDescriptor;
import org.mule.runtime.module.embedded.api.Product.ArtifactCoordinates;
import org.mule.runtime.module.embedded.api.dependencies.DependencyResolver;

import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.ArrayList;
import java.util.List;
import java.util.Properties;
import java.util.Set;

import org.slf4j.Logger;

public class EmbeddedImplementationUtils {

  private static final Logger LOGGER = getLogger(EmbeddedImplementationUtils.class);

  private static final String IGNORED_DEPENDENCIES_FILE_NAME = "ignored-dependencies.properties";

  private static URL getEmbeddedImplUrl(DependencyResolver dependencyResolver, String muleVersion) throws MalformedURLException {
    BundleDescriptor embeddedImplDescriptor = new BundleDescriptor.Builder().setGroupId("org.mule.distributions")
        .setArtifactId("mule-module-embedded-impl").setVersion(muleVersion).setType("jar").build();

    return dependencyResolver.resolveBundleDescriptor(embeddedImplDescriptor).getBundleUri().toURL();
  }

  public static ClassLoader createEmbeddedImplClassLoader(ClassLoader parentClassLoader, DependencyResolver dependencyResolver,
                                                          String muleVersion)
      throws MalformedURLException {
    BundleDescriptor embeddedBomDescriptor = new BundleDescriptor.Builder().setGroupId("org.mule.distributions")
        .setArtifactId("mule-module-embedded-impl-bom").setVersion(muleVersion).setType("pom").build();

    List<BundleDependency> embeddedImplDependencies =
        dependencyResolver.resolveBundleDescriptorDependencies(embeddedBomDescriptor);

    List<URL> embeddedUrls = embeddedImplDependencies.stream()
        .filter(bundleDependency -> !bundleDependency.getScope().equals(PROVIDED))
        .map(dep -> {
          try {
            return dep.getBundleUri().toURL();
          } catch (MalformedURLException e) {
            throw new RuntimeException(e);
          }
        })
        .collect(toList());
    embeddedUrls.add(getEmbeddedImplUrl(dependencyResolver, muleVersion));

    return new URLClassLoader(embeddedUrls.toArray(new URL[embeddedUrls.size()]), parentClassLoader);
  }

  public static List<ArtifactCoordinates> getDependenciesToIgnoreForCurrentJavaVersion(DependencyResolver dependencyResolver,
                                                                                       String muleVersion) {
    String currentJavaVersion = getProperty("java.specification.version");
    try {
      List<ArtifactCoordinates> dependenciesToIgnore = new ArrayList<>();

      getDependenciesToIgnore(dependencyResolver, muleVersion).forEach((dependency, javaVersions) -> {
        Set<String> javaVersionsSet = stream(((String) javaVersions).split(",")).collect(toSet());
        if (javaVersionsSet.stream().anyMatch(currentJavaVersion::equals)) {
          String groupId = ((String) dependency).split(":")[0];
          String artifactId = ((String) dependency).split(":")[1];
          dependenciesToIgnore.add(new ArtifactCoordinates(groupId, artifactId));
        }
      });

      LOGGER.info("Dependencies to ignore for Runtime version {} and Java version {}: {}", muleVersion, currentJavaVersion,
                  dependenciesToIgnore);

      return dependenciesToIgnore;
    } catch (Exception e) {
      LOGGER.debug("Couldn't retrieve dependencies to ignore for Runtime version {} and Java version {}", muleVersion,
                   currentJavaVersion, e);

      // Java 11+ already contains the packages we need from this jar (org.w3c.dom.*),
      // so it is removed to avoid conflicts with the classes provided by the jdk.
      ArtifactCoordinates xmlApis = new ArtifactCoordinates("xml-apis", "xml-apis");
      return isJavaVersionAtLeast(JAVA_17) ? singletonList(xmlApis) : emptyList();
    }
  }

  public static Properties getDependenciesToIgnore(DependencyResolver dependencyResolver, String muleVersion) throws Exception {
    try (URLClassLoader classLoader = new URLClassLoader(new URL[] {getEmbeddedImplUrl(dependencyResolver, muleVersion)})) {
      Properties dependenciesToIgnoreProperties = new Properties();
      dependenciesToIgnoreProperties.load(classLoader.getResourceAsStream(IGNORED_DEPENDENCIES_FILE_NAME));

      return dependenciesToIgnoreProperties;
    }
  }

}
