/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.embedded.internal.classloading;

import static java.util.Arrays.asList;
import static java.util.Collections.emptySet;
import static java.util.Collections.singleton;
import static java.util.stream.Collectors.toSet;
import static java.util.stream.Stream.concat;

import static org.mule.runtime.module.embedded.internal.classloading.JreExplorer.exploreJdk;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

// TODO MULE-11882 - Consolidate classloading isolation
public class JdkOnlyClassLoaderFactory {

  public static final Set<String> BOOT_PACKAGES = singleton("com.yourkit");
  private static final Set<String> PACKAGES_TO_EXCLUDE =
      new HashSet<>(asList("org.slf4j", "org.apache.logging", "org.eclipse.aether", "org.apache.maven",
                           "org.sonatype.plexus", "org.mule.maven"));

  public static FilteringClassLoader create(ClassLoader parentClassLoader, Set<String> additionalExportedBootPackages) {
    Set<String> packages = new HashSet<>(1024);
    Set<String> exportedResources = new HashSet<>(1024);
    List<ExportedService> services = new ArrayList<>(128);
    exploreJdk(packages, exportedResources, services);
    packages = sanitizeExportedPackages(packages);
    ClassLoaderFilter classLoaderFilter =
        new ClassLoaderFilter(getExportedBootPackages(packages, additionalExportedBootPackages), exportedResources);

    return new FilteringClassLoader(parentClassLoader, classLoaderFilter, services);
  }

  private static Set<String> sanitizeExportedPackages(Set<String> packages) {
    return packages.stream()
        .filter(p -> PACKAGES_TO_EXCLUDE.stream().noneMatch(p::startsWith))
        .collect(toSet());
  }

  public static FilteringClassLoader create(ClassLoader parentClassLoader) {
    return create(parentClassLoader, emptySet());
  }

  private static Set<String> getExportedBootPackages(Set<String> packages, Set<String> additionalExportedBootPackages) {
    return concat(concat(BOOT_PACKAGES.stream(), packages.stream()), additionalExportedBootPackages.stream()).collect(toSet());
  }
}
