/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.embedded.internal;

import static org.mule.runtime.module.embedded.internal.utils.DependenciesUtils.dependencyToUrl;
import static org.mule.runtime.module.embedded.internal.utils.JpmsUtils.createJpmsUtilsClassLoader;

import static java.lang.ClassLoader.getSystemClassLoader;
import static java.util.Arrays.asList;
import static java.util.Collections.singletonList;
import static java.util.Optional.of;

import static org.apache.commons.io.FileUtils.toFile;
import static org.apache.commons.lang3.JavaVersion.JAVA_11;
import static org.apache.commons.lang3.SystemUtils.isJavaVersionAtMost;

import org.mule.maven.client.api.BundleDependenciesResolutionException;
import org.mule.maven.client.api.MavenClient;
import org.mule.maven.pom.parser.api.model.BundleDescriptor;
import org.mule.runtime.module.embedded.api.dependencies.MuleDependenciesResolver;

import java.io.File;
import java.io.IOException;
import java.lang.reflect.Method;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.List;
import java.util.Optional;
import java.util.Properties;

/**
 * Creates a {@link ClassLoader} with the {@link URL}'s for the Container.
 *
 * @since 1.5
 */
public class MavenContainerOptSeparateClassLoaderFactory extends MavenContainerClassLoaderFactory {

  private static final String ARTIFACT_INFO_PROPERTIES = "artifact-info.properties";

  private final MavenClient mavenClient;
  private final RuntimeProduct runtimeProduct;
  private final boolean isolate;

  public MavenContainerOptSeparateClassLoaderFactory(MuleDependenciesResolver muleDependenciesResolver, MavenClient mavenClient,
                                                     RuntimeProduct runtimeProduct, boolean isolate) {
    super(muleDependenciesResolver);
    this.mavenClient = mavenClient;
    this.runtimeProduct = runtimeProduct;
    this.isolate = isolate;
  }

  @Override
  public ClassLoader create(URL containerBaseFolder) {
    try {

      File containerFolderFile = toFile(containerBaseFolder);
      // the URL has to be constructed this way since File.toURI().toURL() gets rid of the final slash
      URL configurationFolderUrl = new URL(new File(containerFolderFile, "conf").toURI() + "/");

      ClassLoader jpmsUtilsClassLoader = getJpmsUtilsClassLoader(getMuleDependenciesResolver().resolveMuleLibs());
      ClassLoader referenceClassLoader = getReferenceClassLoader(runtimeProduct, jpmsUtilsClassLoader);
      URLClassLoader parentClassLoader = new URLClassLoader(singletonList(configurationFolderUrl).toArray(new URL[1]),
                                                            buildParentClassLoader(referenceClassLoader));
      return createModuleLayerClassLoader(getMuleDependenciesResolver().resolveOptLibs().toArray(new URL[0]),
                                          getMuleDependenciesResolver().resolveMuleLibs().toArray(new URL[0]),
                                          parentClassLoader,
                                          of(getReferenceClass(runtimeProduct, referenceClassLoader)),
                                          jpmsUtilsClassLoader);
    } catch (BundleDependenciesResolutionException e) {
      throw new IllegalArgumentException("Could not find embedded container bom artifact", e);
    } catch (MalformedURLException e) {
      throw new RuntimeException(e);
    }
  }

  private ClassLoader createModuleLayerClassLoader(URL[] optUrls, URL[] muleUrls,
                                                   ClassLoader parentClassLoader,
                                                   Optional<Class> clazz,
                                                   ClassLoader jpmsUtilsClassLoader) {
    if (isJavaVersionAtMost(JAVA_11)) {
      return new URLClassLoader(muleUrls, new URLClassLoader(optUrls, parentClassLoader));
    }

    try {
      Class<?> multiLevelClassLoaderFactoryClass =
          jpmsUtilsClassLoader.loadClass("org.mule.runtime.jpms.api.MultiLevelClassLoaderFactory");
      Object multiLevelClassLoaderFactoryImpl =
          multiLevelClassLoaderFactoryClass.getField("MULTI_LEVEL_URL_CLASSLOADER_FACTORY").get(null);
      return createModuleLayerClassLoader(jpmsUtilsClassLoader,
                                          asList(URL[].class, URL[].class, multiLevelClassLoaderFactoryClass, ClassLoader.class,
                                                 Optional.class),
                                          asList((Object) optUrls, (Object) muleUrls, multiLevelClassLoaderFactoryImpl,
                                                 parentClassLoader, clazz));
    } catch (Exception e) {
      throw new RuntimeException(e);
    }
  }

  private ClassLoader createModuleLayerClassLoader(ClassLoader jpmsUtilsClassLoader, List<Class<?>> parameterTypes,
                                                   List<Object> parameters) {
    try {
      Method createModuleLayerClassLoaderMethod = jpmsUtilsClassLoader.loadClass("org.mule.runtime.jpms.api.JpmsUtils")
          .getMethod("createModuleLayerClassLoader", parameterTypes.toArray(new Class[parameterTypes.size()]));
      return (ClassLoader) createModuleLayerClassLoaderMethod.invoke(null, parameters.toArray());
    } catch (Exception e) {
      throw new RuntimeException(e);
    }
  }

  private ClassLoader getJpmsUtilsClassLoader(List<URL> muleLibs) {
    String jpmsUtilsLibName = "mule-module-jpms-utils";
    URL jpmsUtilsLib = muleLibs.stream().filter(url -> url.toExternalForm().contains(jpmsUtilsLibName)).findFirst().orElse(null);
    if (jpmsUtilsLib == null) {
      throw new RuntimeException("Could not find '" + jpmsUtilsLibName + "' among mule libs");
    }

    ClassLoader jpmsUtilsClassLoader = createJpmsUtilsClassLoader(jpmsUtilsLib);
    return jpmsUtilsClassLoader;
  }

  private ClassLoader getReferenceClassLoader(RuntimeProduct runtimeProduct, ClassLoader jpmsUtilsClassLoader) {
    if (runtimeProduct.isSupportsIsolation() && isolate) {
      URL[] embeddedCommonsUrls = {getEmbeddedCommons()};
      ClassLoader parent = getSystemClassLoader().getParent();

      if (isJavaVersionAtMost(JAVA_11)) {
        return new URLClassLoader(embeddedCommonsUrls, parent);
      }

      return createModuleLayerClassLoader(jpmsUtilsClassLoader, asList(URL[].class, ClassLoader.class),
                                          asList((Object) embeddedCommonsUrls, parent));
    }

    return this.getClass().getClassLoader();
  }

  private Class<?> getReferenceClass(RuntimeProduct runtimeProduct, ClassLoader referenceClassLoader) {
    if (runtimeProduct.isSupportsIsolation() && isolate) {
      return getEmbeddedCommonsClass(referenceClassLoader);
    }

    return this.getClass();
  }

  private Class<?> getEmbeddedCommonsClass(ClassLoader commonsClassLoader) {
    try {
      return commonsClassLoader.loadClass("org.mule.runtime.module.embedded.commons.api.ArtifactConfiguration");
    } catch (ClassNotFoundException e) {
      throw new RuntimeException(e);
    }
  }

  private URL getEmbeddedCommons() {
    BundleDescriptor embeddedCommonsBundleDescriptor = new BundleDescriptor.Builder()
        .setGroupId("org.mule.runtime")
        .setArtifactId("mule-embedded-commons")
        .setVersion(getImplementationVersion())
        .setType("jar")
        .build();
    return dependencyToUrl().apply(mavenClient.resolveBundleDescriptor(embeddedCommonsBundleDescriptor));
  }

  private String getImplementationVersion() {
    try {
      Properties artifactInfo = new Properties();
      artifactInfo.load(this.getClass().getClassLoader().getResourceAsStream(ARTIFACT_INFO_PROPERTIES));

      return artifactInfo.getProperty("artifact-version");
    } catch (IOException e) {
      throw new RuntimeException(e);
    }
  }

}
