/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.embedded.internal;

import static java.lang.Thread.currentThread;

import java.lang.reflect.InvocationTargetException;
import java.util.concurrent.Callable;

public class EmbeddedManifest {

  private final Class<?> manifestClass;
  private final Object manifest;

  static EmbeddedManifest getMuleManifest(ClassLoader muleContainerClassLoader) {
    return executeWithinClassLoader(muleContainerClassLoader, () -> {
      try {
        Class<?> manifestClass = muleContainerClassLoader.loadClass("org.mule.runtime.manifest.api.MuleManifest");
        return new EmbeddedManifest(manifestClass, manifestClass.getMethod("getMuleManifest").invoke(null));
      } catch (ClassNotFoundException e) {
        return new EmbeddedManifest(muleContainerClassLoader.loadClass("org.mule.runtime.core.api.config.MuleManifest"));
      } catch (Exception e) {
        throw new RuntimeException(e);
      }
    });
  }

  private static <T> T executeWithinClassLoader(ClassLoader classLoader, Callable<T> task) {
    ClassLoader contextClassLoader = currentThread().getContextClassLoader();
    try {
      currentThread().setContextClassLoader(classLoader);
      return task.call();
    } catch (Exception e) {
      throw new RuntimeException(e);
    } finally {
      currentThread().setContextClassLoader(contextClassLoader);
    }
  }

  EmbeddedManifest(Class<?> manifestClass) {
    this(manifestClass, null);
  }

  EmbeddedManifest(Class<?> manifestClass, Object manifest) {
    this.manifestClass = manifestClass;
    this.manifest = manifest;
  }

  public String getProductVersion() {
    return invokeMethod("getProductVersion");
  }

  public String getSupportedJdks() {
    return invokeMethod("getSupportedJdks");
  }

  public String getRecommendedJdks() {
    if (manifest != null) {
      return invokeMethod("getRecommendedJdks");
    } else {
      return invokeMethod("getRecommndedJdks");
    }
  }

  private String invokeMethod(String methodName) {
    try {
      if (manifest != null) {
        return (String) manifestClass.getDeclaredMethod(methodName).invoke(manifest);
      } else {
        return (String) manifestClass.getDeclaredMethod(methodName).invoke(null);
      }
    } catch (IllegalAccessException | InvocationTargetException | NoSuchMethodException e) {
      throw new RuntimeException(e);
    }
  }

}
