/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.embedded.api.controller;

import org.mule.runtime.module.embedded.api.ArtifactConfiguration;

import java.io.IOException;

/**
 * Controller class for the runtime.
 *
 * @since 1.6
 */
public interface EmbeddedController {

  /**
   * Starts the Mule Container in embedded mode.
   *
   * @throws Exception in case of failure.
   */
  void start() throws Exception;

  /**
   * Deploys an application with the given configuration in the embedded container.
   *
   * @param artifactConfiguration configuration of the application.
   * @throws IOException            if deserializing the configuration fails.
   * @throws ClassNotFoundException if deserializing the configuration fails.
   */
  void deployApplication(ArtifactConfiguration artifactConfiguration);

  /**
   * Undeploys the application with the given name from the embedded container.
   *
   * @param applicationName name of the application to undeploy.
   * @throws IOException            if deserializing the application name fails.
   * @throws ClassNotFoundException if deserializing the application name fails.
   */
  void undeployApplication(String applicationName);

  /**
   * Deploys a domain with the given configuration in the embedded container.
   *
   * @param artifactConfiguration configuration of the domain.
   * @throws IOException            if deserializing the configuration fails.
   * @throws ClassNotFoundException if deserializing the configuration fails.
   */
  void deployDomain(ArtifactConfiguration artifactConfiguration);

  /**
   * Undeploys the domain with the given name from the embedded container.
   *
   * @param domainName name of the domain to undeploy.
   * @throws IOException            if deserializing the domain name fails.
   * @throws ClassNotFoundException if deserializing the domain name fails.
   */
  void undeployDomain(String domainName);

  /**
   * Stops and disposes the embedded container.
   */
  void stop();

}
