/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.core.api.util;

import static org.apache.commons.collections4.MapUtils.getBooleanValue;
import static org.apache.commons.collections4.MapUtils.getDoubleValue;
import static org.apache.commons.collections4.MapUtils.getIntValue;
import static org.apache.commons.collections4.MapUtils.getLongValue;
import static org.apache.commons.collections4.MapUtils.getString;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.collection.IsMapWithSize.anEmptyMap;
import static org.hamcrest.core.Is.is;
import static org.hamcrest.core.IsNull.notNullValue;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;

import org.mule.tck.junit4.AbstractMuleTestCase;
import org.mule.tck.size.SmallTest;

import java.text.NumberFormat;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;

import org.apache.commons.collections4.keyvalue.DefaultMapEntry;
import org.hamcrest.core.IsNull;
import org.junit.jupiter.api.Test;

@SmallTest
class PropertiesUtilsTestCase extends AbstractMuleTestCase {

  @Test
  void testRemoveNameSpacePrefix() {
    String temp = "this.is.a.namespace";
    String result = PropertiesUtils.removeNamespacePrefix(temp);
    assertEquals("namespace", result);

    temp = "this.namespace";
    result = PropertiesUtils.removeNamespacePrefix(temp);
    assertEquals("namespace", result);

    temp = "namespace";
    result = PropertiesUtils.removeNamespacePrefix(temp);
    assertEquals("namespace", result);

    temp = "this_is-a-namespace";
    result = PropertiesUtils.removeNamespacePrefix(temp);
    assertEquals("this_is-a-namespace", result);
  }

  @Test
  void testRemoveXMLNameSpacePrefix() {
    String temp = "j:namespace";
    String result = PropertiesUtils.removeXmlNamespacePrefix(temp);
    assertEquals("namespace", result);

    temp = "this-namespace";
    result = PropertiesUtils.removeNamespacePrefix(temp);
    assertEquals("this-namespace", result);

    temp = "namespace";
    result = PropertiesUtils.removeNamespacePrefix(temp);
    assertEquals("namespace", result);
  }

  @Test
  void testRemoveNamespaces() throws Exception {
    Map props = new HashMap();

    props.put("blah.booleanProperty", "true");
    props.put("blah.blah.doubleProperty", NumberFormat.getInstance().format(0.124));
    props.put("blah.blah.Blah.intProperty", "14");
    props.put("longProperty", "999999999");
    props.put("3456.stringProperty", "string");

    props = PropertiesUtils.removeNamespaces(props);

    assertTrue(getBooleanValue(props, "booleanProperty", false));
    assertEquals(0.124, 0, getDoubleValue(props, "doubleProperty", 0));
    assertEquals(14, getIntValue(props, "intProperty", 0));
    assertEquals(999999999, 0, getLongValue(props, "longProperty", 0));
    assertEquals("string", getString(props, "stringProperty", ""));
  }

  @Test
  void testMaskedProperties() {
    // test nulls
    assertNull(PropertiesUtils.maskedPropertyValue(null));
    assertNull(PropertiesUtils.maskedPropertyValue(new DefaultMapEntry(null, "value")));
    assertNull(PropertiesUtils.maskedPropertyValue(new DefaultMapEntry("key", null)));

    // try non-masked value
    Map.Entry property = new DefaultMapEntry("secretname", "secret");
    assertTrue("secret".equals(PropertiesUtils.maskedPropertyValue(property)));

    // now mask value
    PropertiesUtils.registerMaskedPropertyName("secretname");
    String masked = PropertiesUtils.maskedPropertyValue(property);
    assertFalse("secret".equals(masked));
    assertTrue(masked.startsWith("*"));
  }

  @Test
  void testLoadAllProperties() {
    Properties properties =
        PropertiesUtils.loadAllProperties("META-INF/org/mule/runtime/core/config/test.properties",
                                          this.getClass().getClassLoader());
    assertThat((String) properties.get("java.lang.IllegalArgumentException"), is("104000"));
  }

  @Test
  void testLoadAllPropertiesNoFile() {
    Properties properties = PropertiesUtils.loadAllProperties("META-INF/org/mule/config/mule-non-existent.properties",
                                                              this.getClass().getClassLoader());
    assertThat(properties, IsNull.notNullValue());
    assertThat(properties, anEmptyMap());
  }

  @Test
  void testLoadAllPropertiesEmptyFile() {
    Properties properties =
        PropertiesUtils.loadAllProperties("META-INF/org/mule/runtime/core/config/mule-empty.properties",
                                          this.getClass().getClassLoader());
    assertThat(properties, notNullValue());
    assertThat(properties, anEmptyMap());
  }

  @Test
  void noPropertiesAreFoundOnEmptyQueryString() {
    Properties properties = PropertiesUtils.getPropertiesFromQueryString("");
    assertThat(properties, anEmptyMap());
  }

}
