/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.cfg.internal.queries;

import static org.mule.runtime.cfg.AllureCfgConstants.CFG_FEATURE;
import static org.mule.runtime.cfg.AllureCfgConstants.Cfg.QUERIES;
import static org.mule.runtime.cfg.utils.CfgTestUtils.hasIdentifier;
import static org.mule.runtime.cfg.utils.CfgTestUtils.JMS_PUBLISH;
import static org.mule.runtime.cfg.utils.CfgTestUtils.LOGGER;
import static org.mule.runtime.cfg.utils.CfgTestUtils.NOOP;
import static org.mule.runtime.cfg.utils.CfgTestUtils.NOOP2;
import static org.mule.runtime.cfg.utils.CfgTestUtils.PARALLEL_FOREACH;
import static org.mule.runtime.cfg.utils.CfgTestUtils.SET_VARIABLE;
import static org.mule.runtime.cfg.utils.CfgTestUtils.testTreeWithErrorHandling;
import static org.hamcrest.CoreMatchers.is;
import static org.hamcrest.MatcherAssert.assertThat;

import org.mule.runtime.cfg.api.ChainExecutionPathTree;
import io.qameta.allure.Feature;
import io.qameta.allure.Story;
import org.junit.Before;
import org.junit.Test;

@Feature(CFG_FEATURE)
@Story(QUERIES)
public class ErrorHandlingQueriesTestCase {

  private ChainExecutionPathTree tree;

  @Before
  public void setup() {
    tree = testTreeWithErrorHandling(true);
  }

  @Test
  public void inAnErrorHandler() {
    assertThat(tree.anyExecutionPathContains(tree -> hasIdentifier(tree, JMS_PUBLISH)), is(true));
    assertThat(tree.allExecutionPathsContain(tree -> hasIdentifier(tree, JMS_PUBLISH)), is(false));
    assertThat(tree.noneExecutionPathsContains(tree -> hasIdentifier(tree, JMS_PUBLISH)), is(false));
  }

  @Test
  public void inErrorHandlerButNotIfNoError() {
    assertThat(tree.anyExecutionPathContains(tree -> hasIdentifier(tree, SET_VARIABLE)), is(true));
    assertThat(tree.allExecutionPathsContain(tree -> hasIdentifier(tree, SET_VARIABLE)), is(false));
    assertThat(tree.noneExecutionPathsContains(tree -> hasIdentifier(tree, SET_VARIABLE)), is(false));
  }

  @Test
  public void inAllRoutes() {
    assertThat(tree.anyExecutionPathContains(tree -> hasIdentifier(tree, LOGGER)), is(true));
    assertThat(tree.allExecutionPathsContain(tree -> hasIdentifier(tree, LOGGER)), is(true));
    assertThat(tree.noneExecutionPathsContains(tree -> hasIdentifier(tree, LOGGER)), is(false));
  }

  @Test
  public void notInAllRoutesIfAfterOperationWithPotentialError() {
    assertThat(tree.anyExecutionPathContains(tree -> hasIdentifier(tree, NOOP)), is(true));
    assertThat(tree.allExecutionPathsContain(tree -> hasIdentifier(tree, NOOP)), is(false));
    assertThat(tree.noneExecutionPathsContains(tree -> hasIdentifier(tree, NOOP)), is(false));
  }

  @Test
  public void elementNotPresent() {
    assertThat(tree.anyExecutionPathContains(tree -> hasIdentifier(tree, PARALLEL_FOREACH)), is(false));
    assertThat(tree.allExecutionPathsContain(tree -> hasIdentifier(tree, PARALLEL_FOREACH)), is(false));
    assertThat(tree.noneExecutionPathsContains(tree -> hasIdentifier(tree, PARALLEL_FOREACH)), is(true));
  }

  @Test
  public void elementNotPresentForElementDeclaredInUnexecutedErrorHandler() {
    assertThat(tree.anyExecutionPathContains(tree -> hasIdentifier(tree, NOOP2)), is(false));
    assertThat(tree.allExecutionPathsContain(tree -> hasIdentifier(tree, NOOP2)), is(false));
    assertThat(tree.noneExecutionPathsContains(tree -> hasIdentifier(tree, NOOP2)), is(true));
  }

}
