/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal;

import static org.mule.runtime.api.functional.Either.left;
import static org.mule.runtime.api.functional.Either.right;
import static org.mule.runtime.ast.api.util.MuleAstUtils.hasPropertyPlaceholder;

import static java.util.Objects.isNull;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static java.util.Optional.ofNullable;
import static java.util.stream.Collectors.toList;

import static org.apache.commons.lang3.StringUtils.isNotEmpty;

import org.mule.metadata.api.annotation.EnumAnnotation;
import org.mule.runtime.api.functional.Either;
import org.mule.runtime.api.meta.model.parameter.ParameterGroupModel;
import org.mule.runtime.api.meta.model.parameter.ParameterModel;
import org.mule.runtime.api.util.LazyValue;
import org.mule.runtime.ast.api.ComponentGenerationInformation;
import org.mule.runtime.ast.api.ComponentMetadataAst;
import org.mule.runtime.ast.api.ComponentParameterAst;
import org.mule.runtime.ast.api.ParameterResolutionException;
import org.mule.runtime.ast.api.builder.ComponentAstBuilder;
import org.mule.runtime.ast.internal.builder.PropertiesResolver;
import org.mule.runtime.ast.internal.model.ParameterModelUtils;
import org.mule.runtime.ast.internal.param.ExpressionAwareParameter;
import org.mule.runtime.ast.internal.param.ParamTypeResolvingVisitor;
import org.mule.runtime.extension.api.error.ErrorMapping;

import java.util.List;
import java.util.Optional;
import java.util.function.Supplier;


public class DefaultComponentParameterAst implements ComponentParameterAst, ExpressionAwareParameter {

  private final String rawValue;
  private final ParameterModel model;
  private final ParameterGroupModel groupModel;
  private final ComponentMetadataAst metadata;
  private final ComponentGenerationInformation generationInformation;
  private final boolean allowsExpressionWithoutMarkers;

  private volatile LazyValue<String> resolved;
  private volatile LazyValue<Either<String, Object>> value;

  public DefaultComponentParameterAst(String rawValue, ParameterModel model, ParameterGroupModel parameterGroupModel,
                                      ComponentGenerationInformation generationInformation,
                                      PropertiesResolver propertiesResolver,
                                      ParameterModelUtils parameterModelUtils) {
    this(rawValue, model, parameterGroupModel, null, generationInformation, propertiesResolver, parameterModelUtils);
  }

  public DefaultComponentParameterAst(String rawValue, ParameterModel model, ParameterGroupModel parameterGroupModel,
                                      ComponentMetadataAst metadata,
                                      ComponentGenerationInformation generationInformation,
                                      PropertiesResolver propertiesResolver,
                                      ParameterModelUtils parameterModelUtils) {
    this(rawValue, rawValue, null, model, parameterGroupModel, metadata, generationInformation, propertiesResolver,
         parameterModelUtils);
  }

  public DefaultComponentParameterAst(ComponentAstBuilder complexValue, ParameterModel model,
                                      ParameterGroupModel parameterGroupModel,
                                      ComponentMetadataAst metadata, ComponentGenerationInformation generationInformation,
                                      PropertiesResolver propertiesResolver,
                                      ParameterModelUtils parameterModelUtils) {
    this(null, null, complexValue::build, model, parameterGroupModel, metadata, generationInformation, propertiesResolver,
         parameterModelUtils);
  }

  public DefaultComponentParameterAst(List<ComponentAstBuilder> complexValue, ParameterModel model,
                                      ParameterGroupModel parameterGroupModel,
                                      ComponentMetadataAst metadata, ComponentGenerationInformation generationInformation,
                                      PropertiesResolver propertiesResolver,
                                      ParameterModelUtils parameterModelUtils) {
    this(null, null, () -> complexValue.stream().map(ComponentAstBuilder::build).collect(toList()), model,
         parameterGroupModel, metadata, generationInformation, propertiesResolver, parameterModelUtils);
  }

  public DefaultComponentParameterAst(List<ErrorMapping> complexValue, ParameterModel model,
                                      ParameterGroupModel parameterGroupModel,
                                      ComponentGenerationInformation generationInformation,
                                      PropertiesResolver propertiesResolver,
                                      ParameterModelUtils parameterModelUtils) {
    this(null, null, () -> complexValue, model, parameterGroupModel, null, generationInformation, propertiesResolver,
         parameterModelUtils);
  }

  private DefaultComponentParameterAst(String rawValue, String mappedRawValue, Supplier<Object> complexValue,
                                       ParameterModel model, ParameterGroupModel parameterGroupModel,
                                       ComponentMetadataAst metadata, ComponentGenerationInformation generationInformation,
                                       PropertiesResolver propertiesResolver,
                                       ParameterModelUtils parameterModelUtils) {
    this.rawValue = rawValue;
    this.model = model;
    this.groupModel = parameterGroupModel;
    this.metadata = metadata;
    this.generationInformation = generationInformation;
    this.allowsExpressionWithoutMarkers = parameterModelUtils.allowsExpressionWithoutMarkers(model);

    resetResolvedParam(rawValue, complexValue, propertiesResolver);
    if (complexValue == null && rawValue != null && hasPropertyPlaceholder(rawValue)) {
      propertiesResolver.onMappingFunctionChanged(() -> resetResolvedParam(rawValue, complexValue, propertiesResolver));
    }
  }

  private void resetResolvedParam(String rawValue, Supplier<Object> complexValue, PropertiesResolver propertiesResolver) {
    if (complexValue != null) {
      this.resolved = new LazyValue<>();
      this.value = new LazyValue<>(() -> right(complexValue.get()));
    } else {
      if (rawValue != null && hasPropertyPlaceholder(rawValue)) {
        this.resolved = new LazyValue<>(() -> propertiesResolver.apply(rawValue));
      } else {
        this.resolved = new LazyValue<>(rawValue);
      }

      this.value = new LazyValue<>(() -> {
        String resolvedRawValue = getResolvedRawValue();

        if (isNull(resolvedRawValue)) {
          final Object defaultValue = getModel().getDefaultValue();

          if (defaultValue != null) {
            if (defaultValue instanceof String) {
              resolvedRawValue = (String) defaultValue;
            } else if (getModel().getType().getAnnotation(EnumAnnotation.class).isPresent()) {
              resolvedRawValue = ((Enum<?>) defaultValue).name();
            } else {
              return right(defaultValue);
            }
          }
        }

        return ParamTypeResolvingVisitor.resolveParamValue(this, metadata, resolvedRawValue);
      });
    }
  }

  @Override
  public boolean isExpression(Object value) {
    if (value instanceof String) {
      String trim = ((String) value).trim();

      if (trim.startsWith(DEFAULT_EXPRESSION_PREFIX) && trim.endsWith(DEFAULT_EXPRESSION_SUFFIX)) {
        return true;
      }

      return allowsExpressionWithoutMarkers;
    } else {
      return false;
    }
  }

  @Override
  public Optional<String> extractExpression(Object value) {
    Optional<String> result = empty();
    if (isExpression(value)) {
      String expression = (String) value;
      if (isNotEmpty(expression)) {
        String trimmedText = expression.trim();

        if (trimmedText.startsWith(DEFAULT_EXPRESSION_PREFIX) && trimmedText.endsWith(DEFAULT_EXPRESSION_SUFFIX)) {
          result =
              of(trimmedText.substring(DEFAULT_EXPRESSION_PREFIX.length(),
                                       trimmedText.length() - DEFAULT_EXPRESSION_SUFFIX.length()));
        } else {
          result = of(trimmedText);
        }
      }
    }

    return result;
  }

  @Override
  public <T> Either<String, T> getValue() {
    return (Either<String, T>) value.get();
  }

  @Override
  public <T> Either<String, Either<ParameterResolutionException, T>> getValueOrResolutionError() {
    try {
      return getValue().mapRight(fixedValue -> right((T) fixedValue));
    } catch (ParameterResolutionException e) {
      return right(left(e));
    }
  }

  @Override
  public String getRawValue() {
    return rawValue;
  }

  @Override
  public String getResolvedRawValue() {
    return resolved.get();
  }

  @Override
  public ParameterModel getModel() {
    return model;
  }

  @Override
  public ParameterGroupModel getGroupModel() {
    return groupModel;
  }

  @Override
  public Optional<ComponentMetadataAst> getMetadata() {
    return ofNullable(metadata);
  }

  @Override
  public ComponentGenerationInformation getGenerationInformation() {
    return generationInformation;
  }

  @Override
  public boolean isDefaultValue() {
    return ofNullable(getResolvedRawValue())
        .map(v -> {
          if (getModel() != null && getModel().getDefaultValue() != null) {
            return v.equals(getModel().getDefaultValue());
          } else {
            return false;
          }
        }).orElse(true);
  }

  @Override
  public String toString() {
    return "DefaultComponentParameterAst{" + model.getName() + ": " + rawValue + "}";
  }
}
