/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.test.internal.builder;

import static org.mule.runtime.api.util.MuleSystemProperties.SYSTEM_PROPERTY_PREFIX;
import static org.mule.runtime.ast.api.DependencyResolutionMode.MINIMAL;
import static org.mule.runtime.ast.test.AllureConstants.ArtifactAst.ARTIFACT_AST;
import static org.mule.runtime.ast.test.AllureConstants.ArtifactAst.AstConstruction.ARTIFACT_AST_CONSTRUCTION;

import static java.lang.System.clearProperty;
import static java.lang.System.getProperty;
import static java.lang.System.setProperty;
import static java.util.Arrays.asList;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.collection.IsCollectionWithSize.hasSize;
import static org.hamcrest.collection.IsEmptyCollection.empty;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

import org.mule.runtime.api.component.ComponentIdentifier;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.meta.model.XmlDslModel;
import org.mule.runtime.api.meta.model.parameter.ParameterizedModel;
import org.mule.runtime.ast.api.ArtifactAst;
import org.mule.runtime.ast.api.ComponentMetadataAst;
import org.mule.runtime.ast.api.DependencyResolutionMode;
import org.mule.runtime.ast.api.builder.ArtifactAstBuilder;

import java.util.HashSet;
import java.util.Optional;

import org.junit.After;
import org.junit.Before;
import org.junit.Test;

import io.qameta.allure.Feature;
import io.qameta.allure.Story;

@Feature(ARTIFACT_AST)
@Story(ARTIFACT_AST_CONSTRUCTION)
public class DefaultArtifactAstBuilderTestCase {

  private String previousDependencyResolutionModeValue;

  private ExtensionModel extModelA;
  private ExtensionModel extModelB;

  @Before
  public void before() {
    final XmlDslModel xmlDslModelA = XmlDslModel.builder()
        .setNamespace("http://ns.test/extA")
        .setPrefix("extA")
        .build();
    extModelA = mock(ExtensionModel.class);
    when(extModelA.getName()).thenReturn("extA");
    when(extModelA.getXmlDslModel()).thenReturn(xmlDslModelA);

    final XmlDslModel xmlDslModelB = XmlDslModel.builder()
        .setNamespace("http://ns.test/extB")
        .setPrefix("extB")
        .build();
    extModelB = mock(ExtensionModel.class);
    when(extModelB.getName()).thenReturn("extB");
    when(extModelB.getXmlDslModel()).thenReturn(xmlDslModelB);

    previousDependencyResolutionModeValue = getProperty(SYSTEM_PROPERTY_PREFIX + DependencyResolutionMode.class.getName());
    setProperty(SYSTEM_PROPERTY_PREFIX + DependencyResolutionMode.class.getName(), MINIMAL.name());
  }

  @After
  public void after() {
    if (previousDependencyResolutionModeValue != null) {
      setProperty(SYSTEM_PROPERTY_PREFIX + DependencyResolutionMode.class.getName(), previousDependencyResolutionModeValue);
    } else {
      clearProperty(SYSTEM_PROPERTY_PREFIX + DependencyResolutionMode.class.getName());
    }
  }

  @Test
  public void extensionsDependencyResolutionNoDependencies() {
    final ArtifactAstBuilder builder = ArtifactAstBuilder.builder(new HashSet<>(asList(extModelA, extModelB)), Optional.empty());

    final ArtifactAst artifactAst = builder.build();

    assertThat(artifactAst.dependencies(), empty());
  }

  @Test
  public void extensionsDependencyResolutionJustOneDependency() {
    final ArtifactAstBuilder builder = ArtifactAstBuilder.builder(new HashSet<>(asList(extModelA, extModelB)), Optional.empty());

    builder.addTopLevelComponent()
        .withIdentifier(ComponentIdentifier.builder().namespace("extA").namespaceUri("http://ns.test/extA").name("component")
            .build())
        .withParameterizedModel(mock(ParameterizedModel.class))
        .withMetadata(mock(ComponentMetadataAst.class));

    final ArtifactAst artifactAst = builder.build();

    assertThat(artifactAst.dependencies(), hasSize(1));
    assertThat(artifactAst.dependencies().iterator().next().getName(), is("extA"));
  }
}
