/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.test.api.builder;

import static org.mule.runtime.api.component.ComponentIdentifier.buildFromStringRepresentation;
import static org.mule.runtime.api.meta.model.parameter.ParameterGroupModel.DEFAULT_GROUP_NAME;
import static org.mule.runtime.ast.internal.builder.adapter.MetadataTypeModelAdapter.createParameterizedTypeModelAdapter;
import static org.mule.runtime.ast.test.AllureConstants.ArtifactAst.ARTIFACT_AST;
import static org.mule.runtime.ast.test.AllureConstants.ArtifactAst.AstConstruction.ARTIFACT_AST_CONSTRUCTION;
import static org.mule.runtime.ast.test.api.util.TestUtils.createMockParameterGroup;

import static java.util.Arrays.asList;
import static java.util.Collections.emptyList;
import static java.util.Collections.emptySet;
import static java.util.Collections.singleton;
import static java.util.Optional.empty;
import static java.util.Optional.of;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.not;
import static org.hamcrest.Matchers.nullValue;
import static org.hamcrest.Matchers.sameInstance;
import static org.hamcrest.collection.IsMapWithSize.anEmptyMap;
import static org.junit.Assert.assertThrows;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

import org.mule.metadata.api.ClassTypeLoader;
import org.mule.metadata.api.model.MetadataType;
import org.mule.runtime.api.meta.ExpressionSupport;
import org.mule.runtime.api.meta.MuleVersion;
import org.mule.runtime.api.meta.NamedObject;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.meta.model.ModelProperty;
import org.mule.runtime.api.meta.model.ParameterDslConfiguration;
import org.mule.runtime.api.meta.model.XmlDslModel;
import org.mule.runtime.api.meta.model.deprecated.DeprecationModel;
import org.mule.runtime.api.meta.model.display.DisplayModel;
import org.mule.runtime.api.meta.model.display.LayoutModel;
import org.mule.runtime.api.meta.model.parameter.FieldValueProviderModel;
import org.mule.runtime.api.meta.model.parameter.ParameterGroupModel;
import org.mule.runtime.api.meta.model.parameter.ParameterModel;
import org.mule.runtime.api.meta.model.parameter.ParameterRole;
import org.mule.runtime.api.meta.model.parameter.ParameterizedModel;
import org.mule.runtime.api.meta.model.parameter.ValueProviderModel;
import org.mule.runtime.api.meta.model.source.SourceCallbackModel;
import org.mule.runtime.api.meta.model.source.SourceModel;
import org.mule.runtime.api.meta.model.stereotype.StereotypeModel;
import org.mule.runtime.ast.api.ComponentAst;
import org.mule.runtime.ast.api.ComponentGenerationInformation;
import org.mule.runtime.ast.api.builder.ArtifactAstBuilder;
import org.mule.runtime.ast.api.builder.ComponentAstBuilder;
import org.mule.runtime.ast.internal.DefaultComponentMetadataAst;
import org.mule.runtime.ast.internal.DefaultComponentParameterAst;
import org.mule.runtime.ast.internal.builder.BaseComponentAstBuilder;
import org.mule.runtime.ast.internal.builder.PropertiesResolver;
import org.mule.runtime.ast.internal.model.ParameterModelUtils;
import org.mule.runtime.ast.test.internal.builder.ApplicationModelTypeUtilsTestCase;
import org.mule.runtime.extension.api.declaration.type.ExtensionsTypeLoaderFactory;
import org.mule.runtime.extension.api.dsl.syntax.DslElementSyntax;

import java.util.List;
import java.util.Optional;
import java.util.OptionalInt;
import java.util.Set;

import org.junit.Before;
import org.junit.Test;

import io.qameta.allure.Feature;
import io.qameta.allure.Issue;
import io.qameta.allure.Story;

@Feature(ARTIFACT_AST)
@Story(ARTIFACT_AST_CONSTRUCTION)
public class ComponentAstBuilderTestCase {

  private static final String FLAT_PARAM_NAME = "flat";
  private static final String INGROUP_PARAM_NAME = "inGroup";

  private ParameterModel flatParameterModel;
  private ParameterModel inGroupParameterModel;
  private ParameterModel inGroupCollidingParameterModel;

  private ParameterGroupModel showInDslGroup;
  private ParameterGroupModel showInDslCollidingGroup;
  private ParameterGroupModel defaultGroup;

  private ParameterizedModel parameterized;
  private SourceModel sourceParameterized;

  @Before
  public void before() {
    final ClassTypeLoader typeLoader = ExtensionsTypeLoaderFactory.getDefault()
        .createTypeLoader(ApplicationModelTypeUtilsTestCase.class.getClassLoader());
    final MetadataType stringType = typeLoader.load(String.class);

    flatParameterModel = mock(ParameterModel.class);
    when(flatParameterModel.getName()).thenReturn(FLAT_PARAM_NAME);
    when(flatParameterModel.getType()).thenReturn(stringType);

    inGroupParameterModel = new NonMockParameterModel(INGROUP_PARAM_NAME, stringType);

    defaultGroup = mock(ParameterGroupModel.class);
    when(defaultGroup.isShowInDsl()).thenReturn(false);
    when(defaultGroup.getName()).thenReturn("Default");
    when(defaultGroup.getParameterModels()).thenReturn(asList(flatParameterModel));
    when(defaultGroup.getParameter(FLAT_PARAM_NAME)).thenReturn(of(flatParameterModel));

    showInDslGroup = mock(ParameterGroupModel.class);
    when(showInDslGroup.isShowInDsl()).thenReturn(true);
    when(showInDslGroup.getName()).thenReturn("Show in DSL");
    when(showInDslGroup.getParameterModels()).thenReturn(asList(inGroupParameterModel));
    when(showInDslGroup.getParameter(INGROUP_PARAM_NAME)).thenReturn(of(inGroupParameterModel));

    parameterized = mock(ParameterizedModel.class);
    when(parameterized.getAllParameterModels()).thenReturn(asList(flatParameterModel, inGroupParameterModel));
    when(parameterized.getParameterGroupModels()).thenReturn(asList(defaultGroup, showInDslGroup));

    inGroupCollidingParameterModel = new NonMockParameterModel(INGROUP_PARAM_NAME, stringType);

    showInDslCollidingGroup = mock(ParameterGroupModel.class);
    when(showInDslCollidingGroup.isShowInDsl()).thenReturn(true);
    when(showInDslCollidingGroup.getName()).thenReturn("Show in DSL Colliding");
    when(showInDslCollidingGroup.getParameterModels()).thenReturn(asList(inGroupCollidingParameterModel));
    when(showInDslCollidingGroup.getParameter(INGROUP_PARAM_NAME)).thenReturn(of(inGroupCollidingParameterModel));

    sourceParameterized = mock(SourceModel.class);

    final SourceCallbackModel successCallback = mock(SourceCallbackModel.class);
    when(successCallback.getAllParameterModels()).thenReturn(asList(inGroupParameterModel));
    when(successCallback.getParameterGroupModels()).thenReturn(asList(showInDslGroup));

    final SourceCallbackModel errorCallback = mock(SourceCallbackModel.class);
    when(errorCallback.getAllParameterModels()).thenReturn(asList(inGroupCollidingParameterModel));
    when(errorCallback.getParameterGroupModels()).thenReturn(asList(showInDslCollidingGroup));

    when(sourceParameterized.getSuccessCallback()).thenReturn(of(successCallback));
    when(sourceParameterized.getErrorCallback()).thenReturn(of(errorCallback));

    when(sourceParameterized.getAllParameterModels())
        .thenReturn(asList(flatParameterModel, inGroupParameterModel, inGroupCollidingParameterModel));
    when(sourceParameterized.getParameterGroupModels()).thenReturn(asList(defaultGroup));
  }

  @Test
  public void lightNoMetadata() {
    final ComponentAst comp = ComponentAstBuilder.builder().build();

    assertThat(comp.getMetadata(), not(nullValue()));
    assertThat(comp.getMetadata().getParserAttributes(), is(anEmptyMap()));
    assertThat(comp.getMetadata().getDocAttributes(), is(anEmptyMap()));
    assertThat(comp.getMetadata().getFileName(), is(empty()));
    assertThat(comp.getMetadata().getSourceCode(), is(empty()));
    assertThat(comp.getMetadata().getStartColumn(), is(OptionalInt.empty()));
    assertThat(comp.getMetadata().getEndColumn(), is(OptionalInt.empty()));
    assertThat(comp.getMetadata().getStartLine(), is(OptionalInt.empty()));
    assertThat(comp.getMetadata().getEndLine(), is(OptionalInt.empty()));
  }

  @Test
  public void noMetadata() {
    assertThrows("metadata", NullPointerException.class, () -> ArtifactAstBuilder.builder(emptySet(), null).addTopLevelComponent()
        .withIdentifier(buildFromStringRepresentation("ns:comp"))
        .build());
  }

  @Test
  public void withSimpleParamInFlatGroup() {
    final BaseComponentAstBuilder componentBuilder = (BaseComponentAstBuilder) ArtifactAstBuilder.builder(emptySet(), null)
        .addTopLevelComponent()
        .withIdentifier(buildFromStringRepresentation("ns:comp"))
        .withMetadata(DefaultComponentMetadataAst.builder().build());

    componentBuilder.withParameter(flatParameterModel, defaultGroup,
                                   new DefaultComponentParameterAst("value", flatParameterModel, createMockParameterGroup(),
                                                                    mock(ComponentGenerationInformation.class),
                                                                    new PropertiesResolver(),
                                                                    new ParameterModelUtils()),
                                   empty());
    componentBuilder.withParameterizedModel(parameterized);
    componentBuilder.getGenerationInformation().withSyntax(mock(DslElementSyntax.class));
    final ComponentAst comp = componentBuilder.build();

    assertThat(comp.getParameter(DEFAULT_GROUP_NAME, FLAT_PARAM_NAME).getRawValue(), is("value"));
    assertThat(comp.getParameter(DEFAULT_GROUP_NAME, FLAT_PARAM_NAME).getValue().getRight(), is("value"));
  }

  @Test
  public void withSimpleParamInDslGroup() {
    final BaseComponentAstBuilder componentBuilder = (BaseComponentAstBuilder) ArtifactAstBuilder.builder(emptySet(), null)
        .addTopLevelComponent()
        .withIdentifier(buildFromStringRepresentation("ns:comp"))
        .withMetadata(DefaultComponentMetadataAst.builder().build());

    componentBuilder
        .withParameter(inGroupParameterModel, showInDslGroup,
                       new DefaultComponentParameterAst("value", inGroupParameterModel, showInDslGroup,
                                                        mock(ComponentGenerationInformation.class),
                                                        new PropertiesResolver(),
                                                        new ParameterModelUtils()),
                       empty());
    componentBuilder.withParameterizedModel(parameterized);
    componentBuilder.getGenerationInformation().withSyntax(mock(DslElementSyntax.class));
    final ComponentAst comp = componentBuilder.build();

    assertThat(comp.getParameter("Show in DSL", INGROUP_PARAM_NAME).getRawValue(), is("value"));
    assertThat(comp.getParameter("Show in DSL", INGROUP_PARAM_NAME).getValue().getRight(), is("value"));
  }

  @Test
  @Issue("MULE-18759")
  public void sourceParamCallbacksDisambiguation() {
    final BaseComponentAstBuilder componentBuilder = (BaseComponentAstBuilder) ArtifactAstBuilder.builder(emptySet(), null)
        .addTopLevelComponent()
        .withIdentifier(buildFromStringRepresentation("ns:comp"))
        .withMetadata(DefaultComponentMetadataAst.builder().build());

    componentBuilder.withParameter(inGroupParameterModel, showInDslGroup,
                                   new DefaultComponentParameterAst("value", inGroupParameterModel, showInDslGroup,
                                                                    mock(ComponentGenerationInformation.class),
                                                                    new PropertiesResolver(),
                                                                    new ParameterModelUtils()),
                                   empty());
    componentBuilder.withParameter(inGroupCollidingParameterModel, showInDslCollidingGroup,
                                   new DefaultComponentParameterAst("valueColliding", inGroupCollidingParameterModel,
                                                                    showInDslCollidingGroup,
                                                                    mock(ComponentGenerationInformation.class),
                                                                    new PropertiesResolver(),
                                                                    new ParameterModelUtils()),
                                   empty());
    componentBuilder.withParameterizedModel(sourceParameterized);

    componentBuilder.getGenerationInformation().withSyntax(mock(DslElementSyntax.class));

    final ComponentAst comp = componentBuilder.build();

    assertThat(comp.getParameter("Show in DSL", INGROUP_PARAM_NAME).getRawValue(), is("value"));
    assertThat(comp.getParameter("Show in DSL", INGROUP_PARAM_NAME).getValue().getRight(), is("value"));
    assertThat(comp.getParameter("Show in DSL Colliding", INGROUP_PARAM_NAME).getRawValue(), is("valueColliding"));
    assertThat(comp.getParameter("Show in DSL Colliding", INGROUP_PARAM_NAME).getValue().getRight(), is("valueColliding"));
  }

  @Test(expected = IllegalStateException.class)
  public void duplicateParamSet() {
    final BaseComponentAstBuilder componentBuilder = (BaseComponentAstBuilder) ArtifactAstBuilder.builder(emptySet(), null)
        .addTopLevelComponent()
        .withIdentifier(buildFromStringRepresentation("ns:comp"))
        .withMetadata(DefaultComponentMetadataAst.builder().build());

    componentBuilder.withParameter(flatParameterModel, defaultGroup,
                                   new DefaultComponentParameterAst("value", flatParameterModel, defaultGroup,
                                                                    mock(ComponentGenerationInformation.class),
                                                                    new PropertiesResolver(),
                                                                    new ParameterModelUtils()),
                                   empty());
    componentBuilder.withParameter(flatParameterModel, defaultGroup,
                                   new DefaultComponentParameterAst("value", flatParameterModel, defaultGroup,
                                                                    mock(ComponentGenerationInformation.class),
                                                                    new PropertiesResolver(),
                                                                    new ParameterModelUtils()),
                                   empty());
  }

  @Test
  @Issue("MULE-19156")
  public void extensionModelSet() {
    final ExtensionModel nsExtModel = mock(ExtensionModel.class);
    when(nsExtModel.getName()).thenReturn("NS");
    when(nsExtModel.getXmlDslModel()).thenReturn(XmlDslModel.builder()
        .setPrefix("ns")
        .build());

    final ComponentAst comp = ArtifactAstBuilder.builder(singleton(nsExtModel), null)
        .addTopLevelComponent()
        .withIdentifier(buildFromStringRepresentation("ns:comp"))
        .withMetadata(DefaultComponentMetadataAst.builder().build()).build();

    assertThat(comp.getExtensionModel(), sameInstance(nsExtModel));
  }

  @Test
  @Issue("MULE-19363")
  public void metadataTypeSet() {
    final ExtensionModel nsExtModel = mock(ExtensionModel.class);
    when(nsExtModel.getName()).thenReturn("NS");
    when(nsExtModel.getXmlDslModel()).thenReturn(XmlDslModel.builder()
        .setPrefix("ns")
        .build());

    final MetadataType expectedType = mock(MetadataType.class);

    final ComponentAst comp = ArtifactAstBuilder.builder(singleton(nsExtModel), null)
        .addTopLevelComponent()
        .withIdentifier(buildFromStringRepresentation("ns:comp"))
        .withParameterizedModel(createParameterizedTypeModelAdapter(expectedType, null))
        .withMetadata(DefaultComponentMetadataAst.builder().build()).build();

    assertThat(comp.getType(), sameInstance(expectedType));
  }

  /**
   * This is an actual class instead of a mock because the functionality under test depends on the equals implementation of the
   * actual object (as specified in {@link NamedObject}). Since equals cannot be mocked, an actual class is used.
   */
  private static class NonMockParameterModel implements ParameterModel {

    private final String name;
    private final MetadataType type;

    public NonMockParameterModel(String name, MetadataType type) {
      this.name = name;
      this.type = type;
    }

    @Override
    public String getName() {
      return name;
    }

    @Override
    public MetadataType getType() {
      return type;
    }

    @Override
    public int hashCode() {
      final int prime = 31;
      int result = 1;
      result = prime * result + ((name == null) ? 0 : name.hashCode());
      return result;
    }

    @Override
    public boolean equals(Object obj) {
      if (this == obj) {
        return true;
      }
      if (obj == null) {
        return false;
      }
      if (getClass() != obj.getClass()) {
        return false;
      }
      NonMockParameterModel other = (NonMockParameterModel) obj;
      if (name == null) {
        if (other.name != null) {
          return false;
        }
      } else if (!name.equals(other.name)) {
        return false;
      }
      return true;
    }

    // These methods below are not used by the functionality under test

    @Override
    public String getDescription() {
      return null;
    }

    @Override
    public <T extends ModelProperty> Optional<T> getModelProperty(Class<T> propertyType) {
      return empty();
    }

    @Override
    public Set<ModelProperty> getModelProperties() {
      return emptySet();
    }

    @Override
    public boolean hasDynamicType() {
      return false;
    }

    @Override
    public Optional<DisplayModel> getDisplayModel() {
      return null;
    }

    @Override
    public Optional<DeprecationModel> getDeprecationModel() {
      return null;
    }

    @Override
    public boolean isRequired() {
      return false;
    }

    @Override
    public boolean isOverrideFromConfig() {
      return false;
    }

    @Override
    public ExpressionSupport getExpressionSupport() {
      return null;
    }

    @Override
    public Object getDefaultValue() {
      return null;
    }

    @Override
    public ParameterDslConfiguration getDslConfiguration() {
      return null;
    }

    @Override
    public ParameterRole getRole() {
      return null;
    }

    @Override
    public Optional<LayoutModel> getLayoutModel() {
      return null;
    }

    @Override
    public List<StereotypeModel> getAllowedStereotypes() {
      return null;
    }

    @Override
    public Optional<ValueProviderModel> getValueProviderModel() {
      return null;
    }

    @Override
    public List<FieldValueProviderModel> getFieldValueProviderModels() {
      return emptyList();
    }

    @Override
    public boolean isComponentId() {
      return false;
    }

    @Override
    public Optional<MuleVersion> getMinMuleVersion() {
      return empty();
    }
  }
}
