/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal.validation;

import static org.mule.runtime.ast.api.util.MuleAstUtils.validatorBuilder;
import static org.mule.runtime.ast.test.AllureConstants.ArtifactAst.ARTIFACT_AST;
import static org.mule.runtime.ast.test.AllureConstants.ArtifactAst.Validations.VALIDATIONS;

import static java.lang.Thread.currentThread;
import static java.util.Collections.emptyList;
import static java.util.Collections.singletonList;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.core.Is.is;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.ArgumentMatchers.eq;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.mockStatic;
import static org.mockito.Mockito.never;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

import org.mule.runtime.ast.api.ArtifactAst;
import org.mule.runtime.ast.api.ComponentAst;
import org.mule.runtime.ast.api.validation.ArtifactAstValidator;
import org.mule.runtime.ast.api.validation.ArtifactValidation;
import org.mule.runtime.ast.api.validation.Validation;
import org.mule.runtime.ast.api.validation.ValidationsProvider;

import java.util.List;
import java.util.ServiceLoader;
import java.util.function.Consumer;
import java.util.stream.Stream;

import org.junit.Before;
import org.junit.Test;

import org.mockito.MockedStatic;

import io.qameta.allure.Feature;
import io.qameta.allure.Story;

@Feature(ARTIFACT_AST)
@Story(VALIDATIONS)
public class ArtifactAstValidationTestCase {

  private Validation validation;
  private ArtifactValidation artifactValidation;
  private TestValidationProvider testValidationProvider;

  private ArtifactAst ast;
  private ComponentAst component;

  @Before
  public void setUp() {
    validation = mock(Validation.class);
    artifactValidation = mock(ArtifactValidation.class);

    testValidationProvider = new TestValidationProvider();

    ast = mock(ArtifactAst.class);
    component = mock(ComponentAst.class);
    when(ast.topLevelComponentsStream()).thenAnswer(inv -> Stream.of(component));
  }

  private void configureValidationProviderLoaders(MockedStatic<ServiceLoader> loaderClass) {
    ServiceLoader fromRuntimeLoader = mock(ServiceLoader.class);
    when(fromRuntimeLoader.spliterator()).thenReturn(singletonList(testValidationProvider).spliterator());

    ServiceLoader fromExtensionsLoader = mock(ServiceLoader.class);
    when(fromExtensionsLoader.spliterator()).thenReturn(emptyList().spliterator());

    loaderClass.when(() -> ServiceLoader.load(ValidationsProvider.class))
        .thenReturn(fromRuntimeLoader);
    loaderClass.when(() -> ServiceLoader.load(eq(ValidationsProvider.class), any(ClassLoader.class)))
        .thenReturn(fromExtensionsLoader);
  }

  @Test
  public void validationsCalledOnAst() {
    ArtifactAstValidator validator;
    try (MockedStatic<ServiceLoader> loaderClass = mockStatic(ServiceLoader.class)) {
      configureValidationProviderLoaders(loaderClass);

      validator = validatorBuilder()
          .build();
    }

    assertThat(testValidationProvider.getArtifactRegionClassLoader(), is(currentThread().getContextClassLoader()));
    assertThat(testValidationProvider.isIgnoreParamsWithProperties(), is(false));

    verifyValidationsInvoked(validator);
  }

  @Test
  public void validationsNotApplicableNotCalledOnAst() {
    ArtifactAstValidator validator;
    try (MockedStatic<ServiceLoader> loaderClass = mockStatic(ServiceLoader.class)) {
      configureValidationProviderLoaders(loaderClass);

      validator = validatorBuilder()
          .build();
    }

    assertThat(testValidationProvider.getArtifactRegionClassLoader(), is(currentThread().getContextClassLoader()));
    assertThat(testValidationProvider.isIgnoreParamsWithProperties(), is(false));

    when(validation.applicable()).thenReturn(h -> false);

    validator.validate(ast);

    verify(validation).applicable();
    verify(validation, never()).validateMany(component, ast);
    verify(artifactValidation).validateMany(ast);
  }

  @Test
  public void validationEnricher() {
    final Consumer enricher = mock(Consumer.class);

    ArtifactAstValidator validator;
    try (MockedStatic<ServiceLoader> loaderClass = mockStatic(ServiceLoader.class)) {
      configureValidationProviderLoaders(loaderClass);

      validator = validatorBuilder()
          .withValidationEnricher(enricher)
          .build();
    }

    verify(enricher).accept(testValidationProvider);
    assertThat(testValidationProvider.getArtifactRegionClassLoader(), is(currentThread().getContextClassLoader()));
    assertThat(testValidationProvider.isIgnoreParamsWithProperties(), is(false));

    verifyValidationsInvoked(validator);
  }

  @Test
  public void validationsFilter() {
    ArtifactAstValidator validator;
    try (MockedStatic<ServiceLoader> loaderClass = mockStatic(ServiceLoader.class)) {
      configureValidationProviderLoaders(loaderClass);

      validator = validatorBuilder()
          .withValidationsFilter(v -> false)
          .build();
    }

    assertThat(testValidationProvider.getArtifactRegionClassLoader(), is(currentThread().getContextClassLoader()));
    assertThat(testValidationProvider.isIgnoreParamsWithProperties(), is(false));

    when(validation.applicable()).thenReturn(h -> true);

    validator.validate(ast);

    verify(validation, never()).applicable();
    verify(validation, never()).validateMany(component, ast);
    verify(artifactValidation, never()).validateMany(ast);
  }

  @Test
  public void artifactRegionClassLoader() {
    final ClassLoader artifactRegionClassLoader = new ClassLoader(this.getClass().getClassLoader()) {};

    ArtifactAstValidator validator;
    try (MockedStatic<ServiceLoader> loaderClass = mockStatic(ServiceLoader.class)) {
      configureValidationProviderLoaders(loaderClass);

      validator = validatorBuilder()
          .withArtifactRegionClassLoader(artifactRegionClassLoader)
          .build();
    }

    assertThat(testValidationProvider.getArtifactRegionClassLoader(), is(artifactRegionClassLoader));
    assertThat(testValidationProvider.isIgnoreParamsWithProperties(), is(false));

    verifyValidationsInvoked(validator);
  }

  @Test
  public void ignoreParamsWithProperties() {
    final ClassLoader artifactRegionClassLoader = new ClassLoader(this.getClass().getClassLoader()) {};

    ArtifactAstValidator validator;
    try (MockedStatic<ServiceLoader> loaderClass = mockStatic(ServiceLoader.class)) {
      configureValidationProviderLoaders(loaderClass);

      validator = validatorBuilder()
          .ignoreParamsWithProperties(true)
          .build();
    }

    assertThat(testValidationProvider.getArtifactRegionClassLoader(), is(currentThread().getContextClassLoader()));
    assertThat(testValidationProvider.isIgnoreParamsWithProperties(), is(true));

    verifyValidationsInvoked(validator);
  }

  private void verifyValidationsInvoked(ArtifactAstValidator validator) {
    when(validation.applicable()).thenReturn(h -> true);

    validator.validate(ast);

    verify(validation).applicable();
    verify(validation).validateMany(component, ast);
    verify(artifactValidation).validateMany(ast);
  }

  public class TestValidationProvider implements ValidationsProvider {

    private ClassLoader artifactRegionClassLoader;
    private boolean ignoreParamsWithProperties;

    public ClassLoader getArtifactRegionClassLoader() {
      return artifactRegionClassLoader;
    }

    @Override
    public void setArtifactRegionClassLoader(ClassLoader artifactRegionClassLoader) {
      this.artifactRegionClassLoader = artifactRegionClassLoader;
    }

    public boolean isIgnoreParamsWithProperties() {
      return ignoreParamsWithProperties;
    }

    @Override
    public void setIgnoreParamsWithProperties(boolean ignoreParamsWithProperties) {
      this.ignoreParamsWithProperties = ignoreParamsWithProperties;
    }

    @Override
    public List<Validation> get() {
      return singletonList(validation);
    }

    @Override
    public List<ArtifactValidation> getArtifactValidations() {
      return singletonList(artifactValidation);
    }
  }

}
