/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal.builder;

import org.mule.runtime.api.component.TypedComponentIdentifier;
import org.mule.runtime.api.meta.model.EnrichableModel;
import org.mule.runtime.api.meta.model.ModelProperty;
import org.mule.runtime.api.meta.model.config.ConfigurationModel;
import org.mule.runtime.api.meta.model.connection.ConnectionProviderModel;
import org.mule.runtime.api.meta.model.construct.ConstructModel;
import org.mule.runtime.api.meta.model.source.SourceModel;
import org.mule.runtime.dsl.api.component.config.DefaultComponentLocation;
import org.mule.runtime.dsl.api.component.config.DefaultComponentLocation.DefaultLocationPart;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.net.URI;
import java.util.List;
import java.util.Optional;
import java.util.OptionalInt;

import static java.util.Arrays.asList;
import static java.util.Collections.singletonList;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static java.util.Optional.ofNullable;
import static java.util.stream.Collectors.toList;
import static org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType.CHAIN;
import static org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType.ERROR_HANDLER;
import static org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType.ON_ERROR;
import static org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType.OPERATION;
import static org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType.ROUTE;
import static org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType.ROUTER;
import static org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType.SCOPE;
import static org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType.SOURCE;
import static org.mule.runtime.api.component.TypedComponentIdentifier.builder;
import static org.mule.runtime.api.util.classloader.MuleImplementationLoaderUtils.getMuleImplementationsLoader;
import static org.mule.runtime.api.util.classloader.MuleImplementationLoaderUtils.isResolveMuleImplementationLoadersDynamically;

/**
 * Visitor that setups the {@link DefaultComponentLocation} for all mule components in the artifact configuration.
 *
 * @since 1.0
 */
public final class ComponentLocationVisitor {

  private static final Class<? extends ModelProperty> customLocationPartModelPropertyClass;
  private static final Method customLocationPartModelIsIndexedMethod;
  private static final Method customLocationPartModelGetLocationPartMethod;

  private static final Class<? extends ModelProperty> customLocationPartModelPropertyDeprecatedClass;
  private static final Method customLocationPartModelIsIndexedDeprecatedMethod;
  private static final Method customLocationPartModelGetLocationPartDeprecatedMethod;

  static {
    customLocationPartModelPropertyClass = loadClass("org.mule.runtime.core.internal.extension.CustomLocationPartModelProperty");
    customLocationPartModelIsIndexedMethod = getMethod("isIndexed", customLocationPartModelPropertyClass);
    customLocationPartModelGetLocationPartMethod = getMethod("getLocationPart", customLocationPartModelPropertyClass);

    customLocationPartModelPropertyDeprecatedClass =
        loadClass("org.mule.runtime.module.extension.api.loader.java.property.CustomLocationPartModelProperty");
    customLocationPartModelIsIndexedDeprecatedMethod = getMethod("isIndexed", customLocationPartModelPropertyDeprecatedClass);
    customLocationPartModelGetLocationPartDeprecatedMethod =
        getMethod("getLocationPart", customLocationPartModelPropertyDeprecatedClass);
  }

  private static Method getMethod(String name, Class<? extends ModelProperty> clazz) {
    if (clazz == null) {
      return null;
    }

    try {
      return clazz.getMethod(name);
    } catch (NoSuchMethodException e) {
      return null;
    }
  }

  private static Class<? extends ModelProperty> loadClass(String name) {
    Class<? extends ModelProperty> foundClass = null;
    try {
      foundClass = (Class<? extends ModelProperty>) getMuleImplementationsLoader().loadClass(name);
    } catch (ClassNotFoundException | SecurityException e) {
      // No custom location processing
    }

    return foundClass;
  }

  private static Method getCustomLocationPartModelIsIndexedMethod(Class<? extends ModelProperty> modelPropertyClass) {
    if (isResolveMuleImplementationLoadersDynamically()) {
      return getMethod("isIndexed", modelPropertyClass);
    } else {
      return customLocationPartModelIsIndexedMethod;
    }
  }

  private static Method getCustomLocationPartModelGetLocationPartMethod(Class<? extends ModelProperty> modelPropertyClass) {
    if (isResolveMuleImplementationLoadersDynamically()) {
      return getMethod("getLocationPart", modelPropertyClass);
    } else {
      return customLocationPartModelGetLocationPartMethod;
    }
  }

  private static Class<? extends ModelProperty> getCustomLocationPartModelPropertyClass() {
    if (isResolveMuleImplementationLoadersDynamically()) {
      return loadClass("org.mule.runtime.core.internal.extension.CustomLocationPartModelProperty");
    } else {
      return customLocationPartModelPropertyClass;
    }
  }

  private static Method getCustomLocationPartModelIsIndexedDeprecatedMethod(Class<? extends ModelProperty> modelPropertyClass) {
    if (isResolveMuleImplementationLoadersDynamically()) {
      return getMethod("isIndexed", modelPropertyClass);
    } else {
      return customLocationPartModelIsIndexedDeprecatedMethod;
    }
  }

  private static Method getCustomLocationPartModelGetLocationPartDeprecatedMethod(Class<? extends ModelProperty> modelPropertyClass) {
    if (isResolveMuleImplementationLoadersDynamically()) {
      return getMethod("getLocationPart", modelPropertyClass);
    } else {
      return customLocationPartModelGetLocationPartDeprecatedMethod;
    }
  }

  private static Class<? extends ModelProperty> getCustomLocationPartModelPropertyDeprecatedClass() {
    if (isResolveMuleImplementationLoadersDynamically()) {
      return loadClass("org.mule.runtime.module.extension.api.loader.java.property.CustomLocationPartModelProperty");
    } else {
      return customLocationPartModelPropertyDeprecatedClass;
    }
  }

  private final Class<? extends ModelProperty> modelPropertyClass = getCustomLocationPartModelPropertyClass();
  // W-13216278 Added for backwards compatibility, MUnit uses this outside of the runtime
  private final Class<? extends ModelProperty> deprecatedModelPropertyClass = getCustomLocationPartModelPropertyDeprecatedClass();

  public ComponentLocationVisitor() {
    // Nothing to do
  }

  private static final String PROCESSORS_PART_NAME = "processors";
  private static final String SOURCE_PART_NAME = "source";

  public void resolveLocation(DefaultComponentAstBuilder componentModel, List<DefaultComponentAstBuilder> hierarchy) {
    DefaultComponentLocation componentLocation;
    Optional<TypedComponentIdentifier> typedComponentIdentifier =
        of(builder().identifier(componentModel.getIdentifier()).type(componentModel.getComponentType())
            .build());

    if (hierarchy.isEmpty()) {
      String componentModelNameAttribute = componentModel.getComponentId().orElse(null);
      List<DefaultLocationPart> parts =
          asList(new DefaultLocationPart(componentModelNameAttribute,
                                         typedComponentIdentifier,
                                         componentModel.getMetadata().getFileName(),
                                         componentModel.getMetadata().getStartLine(),
                                         componentModel.getMetadata().getStartColumn()));
      componentLocation = new DefaultComponentLocation(ofNullable(componentModelNameAttribute), parts,
                                                       obtainImportChainUris(componentModel));
    } else if (existsWithinRootContainer(hierarchy)) {
      DefaultComponentAstBuilder parentComponentModel = hierarchy.get(hierarchy.size() - 1);

      DefaultComponentLocation parentComponentLocation = (DefaultComponentLocation) parentComponentModel.getLocation();

      Optional<DefaultComponentLocation> locationWithCustomPart =
          resolveLocationWithCustomPart(componentModel, typedComponentIdentifier, parentComponentLocation);

      componentLocation = locationWithCustomPart
          .orElseGet(() -> resolveNestedLocation(componentModel, hierarchy, typedComponentIdentifier, parentComponentModel,
                                                 parentComponentLocation));
    } else {
      DefaultComponentAstBuilder parentComponentModel = hierarchy.get(hierarchy.size() - 1);
      DefaultComponentLocation parentComponentLocation = (DefaultComponentLocation) parentComponentModel.getLocation();
      componentLocation =
          parentComponentLocation.appendLocationPart(findNonProcessorPath(componentModel), typedComponentIdentifier,
                                                     componentModel.getMetadata().getFileName(),
                                                     componentModel.getMetadata().getStartLine(),
                                                     componentModel.getMetadata().getStartColumn());
    }
    componentModel.withLocation(componentLocation);
  }

  private static DefaultComponentLocation resolveNestedLocation(DefaultComponentAstBuilder componentModel,
                                                                List<DefaultComponentAstBuilder> hierarchy,
                                                                Optional<TypedComponentIdentifier> typedComponentIdentifier,
                                                                DefaultComponentAstBuilder parentComponentModel,
                                                                DefaultComponentLocation parentComponentLocation) {
    if (componentModel.getComponentType().equals(CHAIN)) {
      DefaultLocationPart newLastPart = new DefaultLocationPart(parentComponentModel.getComponentId().orElse(null),
                                                                typedComponentIdentifier,
                                                                componentModel.getMetadata().getFileName(),
                                                                componentModel.getMetadata().getStartLine(),
                                                                componentModel.getMetadata().getStartColumn());


      return new DefaultComponentLocation(parentComponentModel.getComponentId(),
                                          singletonList(newLastPart),
                                          obtainImportChainUris(componentModel));
    } else if (hierarchy.size() == 1 && isRootProcessorScope(parentComponentModel)) {
      return processFlowDirectChild(componentModel, hierarchy, parentComponentLocation, typedComponentIdentifier);
    } else if (isErrorHandler(componentModel)) {
      return processErrorHandlerComponent(componentModel, parentComponentLocation, typedComponentIdentifier);
    } else if (isTemplateOnErrorHandler(componentModel)) {
      return processOnErrorModel(componentModel, hierarchy, parentComponentLocation, typedComponentIdentifier);
    } else if (parentComponentIsRouter(hierarchy)) {
      return processRouterChild(componentModel, hierarchy, typedComponentIdentifier, parentComponentLocation);
    } else if (isProcessor(componentModel) && !existsWithinSource(hierarchy)) {
      return parentComponentLocation
          .appendProcessorsPart()
          .appendLocationPart(findProcessorPath(componentModel, hierarchy),
                              typedComponentIdentifier,
                              componentModel.getMetadata().getFileName(),
                              componentModel.getMetadata().getStartLine(),
                              componentModel.getMetadata().getStartColumn());
    } else if (isConnection(componentModel)) {
      return parentComponentLocation.appendConnectionPart(typedComponentIdentifier,
                                                          componentModel.getMetadata().getFileName(),
                                                          componentModel.getMetadata().getStartLine(),
                                                          componentModel.getMetadata().getStartColumn());
    } else {
      return parentComponentLocation.appendLocationPart(findNonProcessorPath(componentModel),
                                                        typedComponentIdentifier,
                                                        componentModel.getMetadata().getFileName(),
                                                        componentModel.getMetadata().getStartLine(),
                                                        componentModel.getMetadata().getStartColumn());
    }
  }

  private Optional<DefaultComponentLocation> resolveLocationWithCustomPart(DefaultComponentAstBuilder componentModel,
                                                                           Optional<TypedComponentIdentifier> typedComponentIdentifier,
                                                                           DefaultComponentLocation parentComponentLocation) {
    Optional<DefaultComponentLocation> locationWithCustomPart = Optional.empty();
    if (getModelPropertyClass() != null) {
      locationWithCustomPart = getLocationWithCustomModel(componentModel, typedComponentIdentifier, parentComponentLocation);
    }

    if (getDeprecatedModelPropertyClass() != null && !locationWithCustomPart.isPresent()) {
      locationWithCustomPart =
          getLocationWithDeprecatedCustomModel(componentModel, typedComponentIdentifier, parentComponentLocation);
    }
    return locationWithCustomPart;
  }

  private Optional<DefaultComponentLocation> getLocationWithDeprecatedCustomModel(DefaultComponentAstBuilder componentModel,
                                                                                  Optional<TypedComponentIdentifier> typedComponentIdentifier,
                                                                                  DefaultComponentLocation parentComponentLocation) {
    Optional<DefaultComponentLocation> locationWithCustomPart;
    locationWithCustomPart = componentModel.getModel(EnrichableModel.class)
        .flatMap(em -> em.getModelProperty(getDeprecatedModelPropertyClass()))
        .flatMap(customLocationPart -> {
          try {
            if (isDeprecatedModelIndexed(customLocationPart)) {
              return of(parentComponentLocation
                  .appendLocationPart(getDeprecatedLocationPath(customLocationPart),
                                      empty(), empty(), OptionalInt.empty(), OptionalInt.empty())
                  .appendLocationPart(findNonProcessorPath(componentModel),
                                      typedComponentIdentifier,
                                      componentModel.getMetadata().getFileName(),
                                      componentModel.getMetadata().getStartLine(),
                                      componentModel.getMetadata().getStartColumn()));
            } else {
              return of(parentComponentLocation
                  .appendLocationPart(getDeprecatedLocationPath(customLocationPart),
                                      typedComponentIdentifier,
                                      componentModel.getMetadata().getFileName(),
                                      componentModel.getMetadata().getStartLine(),
                                      componentModel.getMetadata().getStartColumn()));
            }
          } catch (IllegalArgumentException | BadCustomModelPropertyConfigException e) {
            // Nothing to do, continue silently
            return empty();
          }
        });
    return locationWithCustomPart;
  }

  private Optional<DefaultComponentLocation> getLocationWithCustomModel(DefaultComponentAstBuilder componentModel,
                                                                        Optional<TypedComponentIdentifier> typedComponentIdentifier,
                                                                        DefaultComponentLocation parentComponentLocation) {
    Optional<DefaultComponentLocation> locationWithCustomPart;
    locationWithCustomPart = componentModel.getModel(EnrichableModel.class)
        .flatMap(em -> em.getModelProperty(getModelPropertyClass()))
        .flatMap(customLocationPart -> {
          try {
            if (isModelIndexed(customLocationPart)) {
              return of(parentComponentLocation
                  .appendLocationPart(getCustomModelLocationPath(customLocationPart),
                                      empty(), empty(), OptionalInt.empty(), OptionalInt.empty())
                  .appendLocationPart(findNonProcessorPath(componentModel),
                                      typedComponentIdentifier,
                                      componentModel.getMetadata().getFileName(),
                                      componentModel.getMetadata().getStartLine(),
                                      componentModel.getMetadata().getStartColumn()));
            } else {
              return of(parentComponentLocation
                  .appendLocationPart(getCustomModelLocationPath(customLocationPart),
                                      typedComponentIdentifier,
                                      componentModel.getMetadata().getFileName(),
                                      componentModel.getMetadata().getStartLine(),
                                      componentModel.getMetadata().getStartColumn()));
            }
          } catch (IllegalArgumentException | BadCustomModelPropertyConfigException e) {
            // Nothing to do, continue silently
            return empty();
          }
        });
    return locationWithCustomPart;
  }

  private static boolean isConnection(DefaultComponentAstBuilder componentModel) {
    return componentModel.getModel(ConnectionProviderModel.class).isPresent();
  }

  private static boolean isRoute(DefaultComponentAstBuilder componentModel) {
    return componentModel.getComponentType().equals(ROUTE);
  }

  private static boolean isRootProcessorScope(DefaultComponentAstBuilder componentModel) {
    return componentModel.getModel(ConstructModel.class)
        .map(ConstructModel::allowsTopLevelDeclaration)
        .orElse(false);
  }

  private static boolean parentComponentIsRouter(List<DefaultComponentAstBuilder> hierarchy) {
    return existsWithinRouter(hierarchy)
        && isRouter(hierarchy.get(hierarchy.size() - 1));
  }

  private static boolean existsWithinRouter(List<DefaultComponentAstBuilder> hierarchy) {
    return hierarchy.stream().anyMatch(ComponentLocationVisitor::isRouter);
  }

  private static String findNonProcessorPath(DefaultComponentAstBuilder componentModel) {
    return "" + componentModel.getIndexInParent();
  }

  private static String findRoutePath(DefaultComponentAstBuilder componentModel, List<DefaultComponentAstBuilder> hierarchy) {
    return String.valueOf(hierarchy.get(hierarchy.size() - 1)
        .childComponentsStream()
        .filter(ComponentLocationVisitor::isRoute)
        .collect(toList()).indexOf(componentModel));
  }

  private static DefaultComponentLocation processOnErrorModel(DefaultComponentAstBuilder componentModel,
                                                              List<DefaultComponentAstBuilder> hierarchy,
                                                              DefaultComponentLocation parentComponentLocation,
                                                              Optional<TypedComponentIdentifier> typedComponentIdentifier) {
    final int position = hierarchy.get(hierarchy.size() - 1)
        .childComponentsStream()
        .collect(toList()).indexOf(componentModel);

    return parentComponentLocation.appendLocationPart(String.valueOf(position), typedComponentIdentifier,
                                                      componentModel.getMetadata().getFileName(),
                                                      componentModel.getMetadata().getStartLine(),
                                                      componentModel.getMetadata().getStartColumn());
  }

  private static DefaultComponentLocation processRouterChild(DefaultComponentAstBuilder componentModel,
                                                             List<DefaultComponentAstBuilder> hierarchy,
                                                             Optional<TypedComponentIdentifier> typedComponentIdentifier,
                                                             DefaultComponentLocation parentComponentLocation) {
    if (isRoute(componentModel)) {
      return parentComponentLocation
          .appendRoutePart()
          .appendLocationPart(findRoutePath(componentModel, hierarchy), typedComponentIdentifier,
                              componentModel.getMetadata().getFileName(),
                              componentModel.getMetadata().getStartLine(),
                              componentModel.getMetadata().getStartColumn());
    } else if (isProcessor(componentModel)) {
      // this is the case of the routes directly inside the router as with scatter-gather
      return parentComponentLocation
          .appendRoutePart()
          .appendLocationPart(findProcessorPath(componentModel, hierarchy), empty(), empty(),
                              OptionalInt.empty(), OptionalInt.empty())
          .appendProcessorsPart()
          .appendLocationPart("0", typedComponentIdentifier,
                              componentModel.getMetadata().getFileName(),
                              componentModel.getMetadata().getStartLine(),
                              componentModel.getMetadata().getStartColumn());
    } else {
      return parentComponentLocation
          .appendLocationPart(findNonProcessorPath(componentModel),
                              typedComponentIdentifier,
                              componentModel.getMetadata().getFileName(),
                              componentModel.getMetadata().getStartLine(),
                              componentModel.getMetadata().getStartColumn());
    }
  }

  private static DefaultComponentLocation processFlowDirectChild(DefaultComponentAstBuilder componentModel,
                                                                 List<DefaultComponentAstBuilder> hierarchy,
                                                                 DefaultComponentLocation parentComponentLocation,
                                                                 Optional<TypedComponentIdentifier> typedComponentIdentifier) {
    DefaultComponentLocation componentLocation;
    if (isMessageSource(componentModel)) {
      componentLocation =
          parentComponentLocation.appendLocationPart(SOURCE_PART_NAME, typedComponentIdentifier,
                                                     componentModel.getMetadata().getFileName(),
                                                     componentModel.getMetadata().getStartLine(),
                                                     componentModel.getMetadata().getStartColumn());
    } else if (isProcessor(componentModel)) {
      componentLocation = parentComponentLocation
          .appendLocationPart(PROCESSORS_PART_NAME, empty(), empty(),
                              OptionalInt.empty(), OptionalInt.empty())
          .appendLocationPart(findProcessorPath(componentModel, hierarchy), typedComponentIdentifier,
                              componentModel.getMetadata().getFileName(),
                              componentModel.getMetadata().getStartLine(),
                              componentModel.getMetadata().getStartColumn());
    } else if (isErrorHandler(componentModel)) {
      componentLocation =
          processErrorHandlerComponent(componentModel, parentComponentLocation, typedComponentIdentifier);
    } else {
      componentLocation =
          parentComponentLocation.appendLocationPart(findNonProcessorPath(componentModel),
                                                     typedComponentIdentifier,
                                                     componentModel.getMetadata().getFileName(),
                                                     componentModel.getMetadata().getStartLine(),
                                                     componentModel.getMetadata().getStartColumn());
    }
    return componentLocation;
  }

  private static DefaultComponentLocation processErrorHandlerComponent(DefaultComponentAstBuilder componentModel,
                                                                       DefaultComponentLocation parentComponentLocation,
                                                                       Optional<TypedComponentIdentifier> typedComponentIdentifier) {
    return parentComponentLocation.appendLocationPart("errorHandler", typedComponentIdentifier,
                                                      componentModel.getMetadata().getFileName(),
                                                      componentModel.getMetadata().getStartLine(),
                                                      componentModel.getMetadata().getStartColumn());
  }

  private static String findProcessorPath(DefaultComponentAstBuilder componentModel, List<DefaultComponentAstBuilder> hierarchy) {
    return String.valueOf(hierarchy.get(hierarchy.size() - 1)
        .childComponentsStream()
        .filter(ComponentLocationVisitor::isProcessor)
        .collect(toList())
        .indexOf(componentModel));
  }

  private static boolean existsWithinRootContainer(List<DefaultComponentAstBuilder> hierarchy) {
    return hierarchy.stream()
        .anyMatch(p -> p.getModel(ConfigurationModel.class).isPresent()
            || isRootProcessorScope(p));
  }

  private static boolean existsWithinSource(List<DefaultComponentAstBuilder> hierarchy) {
    return hierarchy.stream()
        .anyMatch(p -> p.getModel(SourceModel.class).isPresent());
  }

  public static boolean isProcessor(DefaultComponentAstBuilder componentModel) {
    return componentModel.getComponentType().equals(OPERATION)
        || componentModel.getComponentType().equals(ROUTER)
        || componentModel.getComponentType().equals(SCOPE);
  }

  public static boolean isMessageSource(DefaultComponentAstBuilder componentModel) {
    return componentModel.getComponentType().equals(SOURCE);
  }

  public static boolean isErrorHandler(DefaultComponentAstBuilder componentModel) {
    return componentModel.getComponentType().equals(ERROR_HANDLER);
  }

  public static boolean isTemplateOnErrorHandler(DefaultComponentAstBuilder componentModel) {
    return componentModel.getComponentType().equals(ON_ERROR);
  }

  public static boolean isRouter(DefaultComponentAstBuilder componentModel) {
    return componentModel.getComponentType().equals(ROUTER);
  }

  private static List<URI> obtainImportChainUris(DefaultComponentAstBuilder componentModel) {
    return componentModel.getMetadata().getImportChain().stream()
        .filter(imp -> imp.getMetadata().getFileUri().isPresent())
        .map(imp -> imp.getMetadata().getFileUri().get())
        .collect(toList());
  }

  /*
   * The methods below were added to make the code more testable (so I could check the special override code for model properties.
   * It's all static stuff.) Use a 'spy' to override these methods. They should inline in practice if there's any trouble...
   */
  Class<? extends ModelProperty> getDeprecatedModelPropertyClass() {
    return deprecatedModelPropertyClass;
  }

  Class<? extends ModelProperty> getModelPropertyClass() {
    return modelPropertyClass;
  }

  boolean isModelIndexed(ModelProperty customLocationPart) {
    return getBoolean(getCustomLocationPartModelIsIndexedMethod(getModelPropertyClass()), customLocationPart);
  }

  String getCustomModelLocationPath(ModelProperty customLocationPart) {
    return getString(getCustomLocationPartModelGetLocationPartMethod(getModelPropertyClass()), customLocationPart);
  }

  String getDeprecatedLocationPath(ModelProperty customLocationPart) {
    return getString(getCustomLocationPartModelGetLocationPartDeprecatedMethod(getDeprecatedModelPropertyClass()),
                     customLocationPart);
  }

  boolean isDeprecatedModelIndexed(ModelProperty customLocationPart) {
    return getBoolean(getCustomLocationPartModelIsIndexedDeprecatedMethod(getDeprecatedModelPropertyClass()), customLocationPart);
  }

  static String getString(Method method, ModelProperty customLocationPart) {
    return Optional.ofNullable(method).map(m -> getStringInternal(m, customLocationPart)).orElse(null);
  }

  static Boolean getBoolean(Method method, ModelProperty customLocationPart) {
    return Optional.ofNullable(method).map(m -> getBooleanInternal(m, customLocationPart)).orElse(false);
  }

  private static boolean getBooleanInternal(Method m, ModelProperty customLocationPart) {
    try {
      return (boolean) m.invoke(customLocationPart);
    } catch (IllegalAccessException | InvocationTargetException e) {
      throw new BadCustomModelPropertyConfigException(e);
    }
  }

  private static String getStringInternal(Method m, ModelProperty customLocationPart) {
    try {
      return (String) m.invoke(customLocationPart);
    } catch (IllegalAccessException | InvocationTargetException e) {
      throw new BadCustomModelPropertyConfigException(e);
    }
  }

  static class BadCustomModelPropertyConfigException extends RuntimeException {

    public BadCustomModelPropertyConfigException(Exception e) {
      super(e);
    }
  }
}
