/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.test.internal.builder.adapter;

import static java.util.Arrays.asList;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.collection.IsCollectionWithSize.hasSize;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.spy;
import static org.mockito.Mockito.when;

import static org.mule.runtime.ast.internal.builder.adapter.MetadataTypeModelAdapter.createMetadataTypeModelAdapterWithStereotype;
import static org.mule.runtime.ast.internal.builder.adapter.MetadataTypeModelAdapter.createParameterizedTypeModelAdapter;
import static org.mule.runtime.ast.test.AllureConstants.ArtifactAst.ARTIFACT_AST;
import static org.mule.runtime.ast.test.AllureConstants.ArtifactAst.AstTraversal.PARAMETER_AST;
import static org.mule.runtime.extension.api.declaration.type.annotation.StereotypeTypeAnnotation.fromDefinitions;

import org.mule.metadata.api.ClassTypeLoader;
import org.mule.metadata.api.model.MetadataType;
import org.mule.runtime.api.meta.model.parameter.ParameterGroupModel;
import org.mule.runtime.api.meta.model.parameter.ParameterModel;
import org.mule.runtime.api.meta.model.stereotype.ImmutableStereotypeModel;
import org.mule.runtime.ast.internal.builder.adapter.MetadataTypeModelAdapter;
import org.mule.runtime.ast.internal.model.DefaultExtensionModelHelper;
import org.mule.runtime.extension.api.annotation.param.Parameter;
import org.mule.runtime.extension.api.declaration.type.ExtensionsTypeLoaderFactory;
import org.mule.runtime.extension.api.declaration.type.annotation.StereotypeTypeAnnotation;
import org.mule.runtime.extension.api.stereotype.StereotypeDefinition;

import java.util.Optional;

import org.junit.Before;
import org.junit.Test;

import io.qameta.allure.Feature;
import io.qameta.allure.Story;

@Feature(ARTIFACT_AST)
@Story(PARAMETER_AST)
public class LegacyStereotypeMetadataTypeModelAdapterTestCase {

  private ClassTypeLoader typeLoader;
  private DefaultExtensionModelHelper extModelHelper;

  @Before
  public void before() {
    typeLoader = ExtensionsTypeLoaderFactory.getDefault()
        .createTypeLoader(LegacyStereotypeMetadataTypeModelAdapterTestCase.class.getClassLoader());

    extModelHelper = mock(DefaultExtensionModelHelper.class);
  }

  @Test
  public void metadataTypeWithStereotype() {
    final MetadataType simplePojoType = spy(typeLoader.load(SimplePojo.class));
    final StereotypeTypeAnnotation stereotypeTypeAnnotation = fromDefinitions(asList(SimpleStereotypeDefinition.class));
    stereotypeTypeAnnotation.resolveStereotypes(stCls -> new ImmutableStereotypeModel("SIMPLE", "TEST", null));
    when(simplePojoType.getAnnotation(StereotypeTypeAnnotation.class)).thenReturn(of(stereotypeTypeAnnotation));
    final Optional<MetadataTypeModelAdapter> adapter =
        createMetadataTypeModelAdapterWithStereotype(simplePojoType, extModelHelper);

    assertThat(adapter.isPresent(), is(true));
  }

  @Test
  public void metadataTypeWithoutStereotype() {
    assertThat(createMetadataTypeModelAdapterWithStereotype(typeLoader.load(SimplePojo.class), extModelHelper).isPresent(),
               is(false));
  }

  @Test
  public void objectTypeFieldsAsParams() {
    final MetadataTypeModelAdapter adapter =
        createParameterizedTypeModelAdapter(typeLoader.load(SimplePojo.class), extModelHelper);

    assertThat(adapter.getAllParameterModels(), hasSize(2));
    final ParameterModel param = adapter.getAllParameterModels().get(0);
    assertThat(param.getName(), is("pojoSimpleParam"));
    assertThat(param.getType(), is(typeLoader.load(String.class)));

    final ParameterModel nameParam = adapter.getAllParameterModels().get(1);
    assertThat(nameParam.getName(), is("name"));

    assertThat(adapter.getParameterGroupModels(), hasSize(1));

    final ParameterGroupModel paramGroup = adapter.getParameterGroupModels().get(0);

    assertThat(paramGroup.getParameterModels(), hasSize(2));
    assertThat(paramGroup.getParameter("pojoSimpleParam").get().getName(), is("pojoSimpleParam"));
    assertThat(paramGroup.isShowInDsl(), is(false));
    assertThat(paramGroup.getLayoutModel(), is(empty()));
  }

  public static class SimplePojo {

    @Parameter
    private String pojoSimpleParam;

  }

  public class SimpleStereotypeDefinition implements StereotypeDefinition {

    @Override
    public String getName() {
      return "SIMPLE";
    }

    @Override
    public String getNamespace() {
      return "TEST";
    }
  }
}
