/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.privileged.error;

import static java.lang.String.format;

import org.mule.api.annotation.NoInstantiate;
import org.mule.runtime.api.message.ErrorType;

import java.util.Objects;

/**
 * Default implementation of {@link ErrorType}.
 *
 * @since 4.6
 */
// This is public and api so that DataWeave can get and invoke its methods and not fallback to change the accessibility of its
// fields
@NoInstantiate
public final class DefaultErrorType implements ErrorType {

  private static final long serialVersionUID = -3716206147606234572L;

  private final String identifier;
  private final String namespace;
  private final ErrorType parentErrorType;

  private final String asString;

  public DefaultErrorType(String identifier, String namespace, ErrorType parentErrorType) {
    this.identifier = identifier;
    this.namespace = namespace;
    this.parentErrorType = parentErrorType;

    this.asString = format("%s:%s", namespace, identifier);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getIdentifier() {
    return identifier;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getNamespace() {
    return namespace;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ErrorType getParentErrorType() {
    return parentErrorType;
  }

  @Override
  public String toString() {
    return asString;
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) {
      return true;
    }
    if (o == null || getClass() != o.getClass()) {
      return false;
    }

    DefaultErrorType that = (DefaultErrorType) o;
    return Objects.equals(identifier, that.identifier) &&
        Objects.equals(namespace, that.namespace) &&
        Objects.equals(parentErrorType, that.parentErrorType);
  }

  @Override
  public int hashCode() {
    return Objects.hash(identifier, namespace, parentErrorType);
  }
}
