/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal.builder.adapter;

import static org.mule.runtime.api.meta.ExpressionSupport.SUPPORTED;
import static org.mule.runtime.api.meta.model.ParameterDslConfiguration.builder;
import static org.mule.runtime.api.meta.model.parameter.ParameterRole.BEHAVIOUR;
import static org.mule.runtime.api.meta.model.parameter.ParameterRole.CONTENT;
import static org.mule.runtime.extension.api.util.ExtensionMetadataTypeUtils.acceptsReferences;

import static java.util.Collections.emptyList;
import static java.util.Collections.unmodifiableSet;
import static java.util.Objects.requireNonNull;
import static java.util.Optional.empty;
import static java.util.Optional.ofNullable;

import org.mule.metadata.api.annotation.DefaultValueAnnotation;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.api.model.ObjectFieldType;
import org.mule.runtime.api.meta.ExpressionSupport;
import org.mule.runtime.api.meta.MuleVersion;
import org.mule.runtime.api.meta.model.ModelProperty;
import org.mule.runtime.api.meta.model.ParameterDslConfiguration;
import org.mule.runtime.api.meta.model.deprecated.DeprecationModel;
import org.mule.runtime.api.meta.model.display.DisplayModel;
import org.mule.runtime.api.meta.model.display.LayoutModel;
import org.mule.runtime.api.meta.model.parameter.FieldValueProviderModel;
import org.mule.runtime.api.meta.model.parameter.ParameterModel;
import org.mule.runtime.api.meta.model.parameter.ParameterRole;
import org.mule.runtime.api.meta.model.parameter.ValueProviderModel;
import org.mule.runtime.api.meta.model.stereotype.StereotypeModel;
import org.mule.runtime.ast.api.MetadataTypeObjectFieldTypeAdapter;
import org.mule.runtime.extension.api.declaration.type.annotation.DisplayTypeAnnotation;
import org.mule.runtime.extension.api.declaration.type.annotation.ExpressionSupportAnnotation;
import org.mule.runtime.extension.api.declaration.type.annotation.LayoutTypeAnnotation;
import org.mule.runtime.extension.api.declaration.type.annotation.StereotypeTypeAnnotation;
import org.mule.runtime.extension.api.property.QNameModelProperty;
import org.mule.runtime.internal.meta.model.DefaultTypeAnnotationModelPropertyWrapper;

import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import javax.xml.namespace.QName;

class ObjectFieldTypeAsParameterModelAdapter implements MetadataTypeObjectFieldTypeAdapter, ParameterModel {

  private final ObjectFieldType wrappedFieldType;
  private final Map<Class<? extends ModelProperty>, ModelProperty> modelProperties = new LinkedHashMap<>();
  private final LayoutModel layoutModel;
  private final ParameterDslConfiguration parameterDslConfiguration;

  public ObjectFieldTypeAsParameterModelAdapter(ObjectFieldType wrappedFieldType, Optional<QName> ownerCustomQName) {
    this.wrappedFieldType = wrappedFieldType;

    parameterDslConfiguration = builder()
        .allowsInlineDefinition(true)
        .allowTopLevelDefinition(false)
        .allowsReferences(acceptsReferences(wrappedFieldType)).build();

    ownerCustomQName
        .map(qName -> new QNameModelProperty(new QName(qName.getNamespaceURI(),
                                                       wrappedFieldType.getKey().getName().getLocalPart(), qName.getPrefix())))
        .ifPresent(qnmp -> modelProperties.put(qnmp.getClass(), qnmp));

    this.wrappedFieldType.getAnnotation(DefaultTypeAnnotationModelPropertyWrapper.class)
        .ifPresent(ann -> {
          final ModelProperty mp = ann.asModelProperty();
          modelProperties.put(mp.getClass(), mp);
        });

    Optional<LayoutTypeAnnotation> optionalLayoutTypeAnnotation =
        this.wrappedFieldType.getAnnotation(LayoutTypeAnnotation.class);
    if (optionalLayoutTypeAnnotation.isPresent()) {
      LayoutTypeAnnotation layoutTypeAnnotation = optionalLayoutTypeAnnotation.get();
      LayoutModel.LayoutModelBuilder layoutModelBuilder = LayoutModel.builder();
      if (layoutTypeAnnotation.isText()) {
        layoutModelBuilder.asText();
      }
      if (layoutTypeAnnotation.isPassword()) {
        layoutModelBuilder.asPassword();
      }
      if (layoutTypeAnnotation.isQuery()) {
        layoutModelBuilder.asQuery();
      }
      layoutTypeAnnotation.getOrder().ifPresent(layoutModelBuilder::order);
      layoutTypeAnnotation.getTabName().ifPresent(layoutModelBuilder::tabName);

      layoutModel = layoutModelBuilder.build();
    } else {
      layoutModel = null;
    }
  }

  @Override
  public String getName() {
    return wrappedFieldType.getKey().getName().getLocalPart();
  }

  @Override
  public String getDescription() {
    return wrappedFieldType.getKey().getDescription().orElse(null);
  }

  @Override
  public <T extends ModelProperty> Optional<T> getModelProperty(Class<T> propertyType) {
    requireNonNull(propertyType);
    return ofNullable((T) modelProperties.get(propertyType));
  }

  @Override
  public Set<ModelProperty> getModelProperties() {
    return unmodifiableSet(new LinkedHashSet<>(modelProperties.values()));
  }

  @Override
  public MetadataType getType() {
    return wrappedFieldType.getValue();
  }

  @Override
  public boolean hasDynamicType() {
    return false;
  }

  @Override
  public Optional<DisplayModel> getDisplayModel() {
    return wrappedFieldType.getAnnotation(DisplayTypeAnnotation.class)
        .flatMap(DisplayTypeAnnotation::getPathModel)
        .map(pm -> DisplayModel.builder().path(pm)
            .build());
  }

  @Override
  public Optional<DeprecationModel> getDeprecationModel() {
    return empty();
  }

  @Override
  public boolean isRequired() {
    return wrappedFieldType.isRequired();
  }

  @Override
  public boolean isOverrideFromConfig() {
    return false;
  }

  @Override
  public ExpressionSupport getExpressionSupport() {
    return wrappedFieldType.getAnnotation(ExpressionSupportAnnotation.class)
        .map(ExpressionSupportAnnotation::getExpressionSupport)
        .orElse(SUPPORTED);
  }

  @Override
  public Object getDefaultValue() {
    return wrappedFieldType.getAnnotation(DefaultValueAnnotation.class)
        .map(DefaultValueAnnotation::getValue)
        .orElseGet(() -> wrappedFieldType.getValue().getAnnotation(DefaultValueAnnotation.class)
            .map(DefaultValueAnnotation::getValue)
            .orElse(null));
  }

  @Override
  public ParameterDslConfiguration getDslConfiguration() {
    return parameterDslConfiguration;
  }

  @Override
  public ParameterRole getRole() {
    return getLayoutModel()
        .map(lyt -> lyt.isText() ? CONTENT : BEHAVIOUR)
        .orElse(BEHAVIOUR);
  }

  @Override
  public Optional<LayoutModel> getLayoutModel() {
    return ofNullable(layoutModel);
  }

  @Override
  public List<StereotypeModel> getAllowedStereotypes() {
    return wrappedFieldType.getAnnotation(StereotypeTypeAnnotation.class)
        .map(StereotypeTypeAnnotation::getAllowedStereotypes)
        .orElseGet(() -> wrappedFieldType.getValue().getAnnotation(StereotypeTypeAnnotation.class)
            .map(StereotypeTypeAnnotation::getAllowedStereotypes)
            .orElse(emptyList()));
  }

  @Override
  public Optional<ValueProviderModel> getValueProviderModel() {
    return empty();
  }

  @Override
  public List<FieldValueProviderModel> getFieldValueProviderModels() {
    return emptyList();
  }

  @Override
  public boolean isComponentId() {
    return getName().equals("name");
  }

  @Override
  public boolean isWrapperFor(ObjectFieldType fieldType) {
    return this.wrappedFieldType.equals(fieldType);
  }

  @Override
  public String toString() {
    return "ObjectFieldTypeAsParameterModelAdapter{" + wrappedFieldType.toString() + "}";
  }

  @Override
  public Optional<MuleVersion> getMinMuleVersion() {
    return empty();
  }
}
