/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal;

import static org.mule.runtime.api.component.ComponentIdentifier.buildFromStringRepresentation;
import static org.mule.runtime.api.component.ComponentIdentifier.builder;
import static org.mule.runtime.api.meta.model.parameter.ParameterGroupModel.DEFAULT_GROUP_NAME;
import static org.mule.runtime.ast.api.error.ErrorTypeRepositoryProvider.getCoreErrorTypeRepo;
import static org.mule.runtime.ast.internal.error.ErrorTypeRepositoryBuildingUtils.ERROR_TYPE_PARAM;
import static org.mule.runtime.ast.internal.error.ErrorTypeRepositoryBuildingUtils.addErrorsFromArtifact;
import static org.mule.runtime.ast.internal.error.ErrorTypeRepositoryBuildingUtils.addErrorsFromExtensions;
import static org.mule.runtime.ast.internal.dsl.DslConstants.CORE_PREFIX;

import static java.util.Arrays.asList;
import static java.util.Arrays.stream;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toSet;

import static org.slf4j.LoggerFactory.getLogger;

import org.mule.runtime.api.component.ComponentIdentifier;
import org.mule.runtime.api.exception.ErrorTypeRepository;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.ast.api.ArtifactAst;
import org.mule.runtime.ast.api.ArtifactType;
import org.mule.runtime.ast.api.ComponentAst;
import org.mule.runtime.ast.api.ImportedResource;
import org.mule.runtime.ast.api.error.ErrorTypeRepositoryProvider;
import org.mule.runtime.ast.api.util.AstTraversalDirection;
import org.mule.runtime.ast.api.util.BaseArtifactAst;
import org.mule.runtime.ast.internal.error.CompositeErrorTypeRepository;
import org.mule.runtime.ast.internal.error.DefaultErrorTypeRepository;

import java.util.Collection;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.function.Predicate;
import java.util.function.Supplier;
import java.util.function.UnaryOperator;
import java.util.stream.Stream;

import org.slf4j.Logger;

public class FilteredArtifactAst extends BaseArtifactAst {

  private static final String ON_ERROR_CONTINUE = "on-error-continue";
  private static final String ON_ERROR_PROPAGATE = "on-error-propagate";

  private static final ComponentIdentifier ON_ERROR_CONTINUE_IDENTIFIER =
      builder().namespace(CORE_PREFIX).name(ON_ERROR_CONTINUE).build();
  private static final ComponentIdentifier ON_ERROR_PROPAGATE_IDENTIFIER =
      builder().namespace(CORE_PREFIX).name(ON_ERROR_PROPAGATE).build();

  private static final Logger LOGGER = getLogger(FilteredArtifactAst.class);

  private final ArtifactAst fullArtifactAst;

  private final Predicate<ComponentAst> componentFilter;

  private final List<ComponentAst> topLevelComponents;
  private final Supplier<ErrorTypeRepository> coreErrorTypeRepo;

  public FilteredArtifactAst(ArtifactAst fullArtifactAst, Predicate<ComponentAst> componentFilter) {
    this(fullArtifactAst, componentFilter, ErrorTypeRepositoryProvider::getCoreErrorTypeRepo);
  }

  FilteredArtifactAst(ArtifactAst fullArtifactAst, Predicate<ComponentAst> componentFilter,
                      Supplier<ErrorTypeRepository> coreErrorTypeRepo) {
    this.fullArtifactAst = fullArtifactAst;
    this.componentFilter = componentFilter;
    this.topLevelComponents = fullArtifactAst.topLevelComponentsStream().filter(componentFilter).collect(toList());
    this.coreErrorTypeRepo = coreErrorTypeRepo;
  }

  @Override
  public String getArtifactName() {
    return fullArtifactAst.getArtifactName();
  }

  @Override
  public ArtifactType getArtifactType() {
    return fullArtifactAst.getArtifactType();
  }

  @Override
  public Set<ExtensionModel> dependencies() {
    return fullArtifactAst.dependencies();
  }

  @Override
  public Optional<ArtifactAst> getParent() {
    return fullArtifactAst.getParent();
  }

  @Override
  public Stream<ComponentAst> recursiveStream(AstTraversalDirection direction) {
    // It overrides the parent method because we need to filter the children and not only the top-level components.
    return fullArtifactAst.recursiveStream(direction).filter(componentFilter);
  }

  @Override
  public List<ComponentAst> topLevelComponents() {
    return topLevelComponents;
  }

  @Override
  public void updatePropertiesResolver(UnaryOperator<String> newPropertiesResolver) {
    fullArtifactAst.updatePropertiesResolver(newPropertiesResolver);
  }

  @Override
  public Collection<ImportedResource> getImportedResources() {
    return fullArtifactAst.getImportedResources();
  }

  @Override
  public ErrorTypeRepository getErrorTypeRepository() {
    LOGGER.debug("Creating errorTypeRepository for Artifact '{}'...", this);

    final ErrorTypeRepository errorTypeRepository = new DefaultErrorTypeRepository();

    final Set<String> usedNamespaces = recursiveStream()
        .map(comp -> comp.getIdentifier().getNamespaceUri())
        .collect(toSet());

    addErrorsFromExtensions(fullArtifactAst.dependencies().stream()
        .filter(em -> usedNamespaces.contains(em.getXmlDslModel().getNamespace()))
        .collect(toSet()), errorTypeRepository);
    addErrorsFromArtifact(this, errorTypeRepository);

    // When lazy init deployment is used an error-mapping may not be initialized due to the component that declares it
    // could not be part of the minimal application model. So, whenever we found that scenario we have to create the
    // errorType if not present in the repository already.
    LOGGER.debug("Registering error handlers referenced errorTypes...");
    recursiveStream()
        .filter(comp -> comp.getIdentifier().equals(ON_ERROR_CONTINUE_IDENTIFIER)
            || comp.getIdentifier().equals(ON_ERROR_PROPAGATE_IDENTIFIER))
        .flatMap(comp -> {
          final String value = comp.getParameter(DEFAULT_GROUP_NAME, ERROR_TYPE_PARAM).getResolvedRawValue();
          if (value != null) {
            return stream(value.split(","));
          } else {
            return Stream.empty();
          }
        })
        .map(errorType -> buildFromStringRepresentation(errorType.trim()))
        .filter(errorType -> !errorTypeRepository.lookupErrorType(errorType).isPresent())
        .forEach(errorType -> {
          LOGGER.debug("Registering errorType '{}'", errorType);
          errorTypeRepository.addErrorType(errorType,
                                           coreErrorTypeRepo.get().getAnyErrorType());
        });

    return new CompositeErrorTypeRepository(asList(coreErrorTypeRepo.get(), errorTypeRepository));
  }
}
