/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.api.exception;

import static org.mule.runtime.api.i18n.I18nMessageFactory.createStaticMessage;

import static java.lang.String.format;
import static java.util.Collections.singletonList;
import static java.util.Collections.unmodifiableList;

import org.mule.runtime.api.exception.MuleRuntimeException;
import org.mule.runtime.api.i18n.I18nMessage;
import org.mule.runtime.api.util.Pair;

import java.util.ArrayList;
import java.util.List;

/**
 * Exception thrown when a key could not be resolved.
 *
 * @since 1.1
 */
public class PropertyNotFoundException extends MuleRuntimeException {

  private static final String PLACEHOLDER_PREFIX = "${";
  private static final String PLACEHOLDER_SUFFIX = "}";
  private static final long serialVersionUID = -3570854244058568638L;

  private final List<Pair<String, String>> unresolvedKeys;

  /**
   * Creates a new instance. This constructor must be used when the resolver has no parent and was not able to resolve a key
   *
   * @param resolverKeyPair the resolver descriptor and the key that was not able to resolve.
   */
  public PropertyNotFoundException(Pair<String, String> resolverKeyPair) {
    super(createFailureException(resolverKeyPair));
    unresolvedKeys = singletonList(resolverKeyPair);
  }

  private static I18nMessage createFailureException(Pair<String, String> resolverKeyPair) {
    return createStaticMessage(createMessageForLeakKey(resolverKeyPair));
  }

  private static String createMessageForLeakKey(Pair<String, String> resolverKeyPair) {
    return format("Couldn't find configuration property value for key %s",
                  PLACEHOLDER_PREFIX + resolverKeyPair.getSecond() + PLACEHOLDER_SUFFIX);
  }

  /**
   * Creates a new instance. This constructor must be used when the resolver invoke the parent and it failed because it wasn't
   * able to resolve a key
   *
   * @param propertyNotFoundException exception thrown by the parent resolver.
   * @param resolverKeyPair           the resolver descriptor and the key that was not able to resolve.
   */
  public PropertyNotFoundException(PropertyNotFoundException propertyNotFoundException,
                                   Pair<String, String> resolverKeyPair) {
    super(createFailureException(resolverKeyPair));
    List<Pair<String, String>> keys = new ArrayList<>(propertyNotFoundException.getUnresolvedKeys());
    keys.add(resolverKeyPair);
    unresolvedKeys = unmodifiableList(keys);
  }

  /**
   * @return a list with the resolvers and the keys that were not resolved. The original key not found is the first element and
   *         the last will be the main resolver.
   */
  public List<Pair<String, String>> getUnresolvedKeys() {
    return unresolvedKeys;
  }
}
