/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal.xml.resolver;

import static org.mule.runtime.api.util.IOUtils.getInputStreamWithCacheControl;
import static org.mule.runtime.api.util.classloader.MuleImplementationLoaderUtils.getMuleImplementationsLoader;
import static org.mule.runtime.dsl.internal.util.SchemaMappingsUtils.CUSTOM_SCHEMA_MAPPINGS_LOCATION;

import org.mule.runtime.dsl.internal.util.SchemaMappingsUtils;

import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.xml.sax.EntityResolver;
import org.xml.sax.InputSource;

/**
 * Custom entity resolver based on Spring's schema resolver.
 * <p>
 * This code was moved from mule project
 *
 * @since 1.0
 */
public class MuleCustomEntityResolver implements EntityResolver {

  private static final Logger LOGGER = LoggerFactory.getLogger(MuleCustomEntityResolver.class);

  private final ClassLoader deploymentClassLoader;
  private final ClassLoader muleImplementationsLoader;
  private final Map<String, String> muleSchemaMappings;
  private final Map<String, String> appPluginsSchemaMappings;

  public MuleCustomEntityResolver(ClassLoader deploymentClassLoader) {
    this.muleImplementationsLoader = getMuleImplementationsLoader();
    this.deploymentClassLoader = deploymentClassLoader;

    final SchemaMappingsUtils schemaMappingsUtils = SchemaMappingsUtils.getFor(muleImplementationsLoader);

    this.muleSchemaMappings = getMuleSchemaMappings(schemaMappingsUtils);
    this.appPluginsSchemaMappings = getAppPluginsSchemaMappings(schemaMappingsUtils);
  }

  @Override
  public InputSource resolveEntity(String publicId, String systemId) {
    if (systemId != null) {
      // Runtime takes precedence over plugins, to avoid a misbehaving plugin to override something from the runtime
      InputSource source =
          resolveEntityInClassloader(muleSchemaMappings, publicId, systemId, muleImplementationsLoader);

      if (source == null) {
        source = resolveEntityInClassloader(appPluginsSchemaMappings, publicId, systemId, this.deploymentClassLoader);
      }

      return source;
    }
    return null;
  }

  private static InputSource resolveEntityInClassloader(Map<String, String> schemaMappings, String publicId, String systemId,
                                                        ClassLoader cl) {
    String resourceLocation = schemaMappings.get(systemId);
    if (resourceLocation != null) {
      URL resource = cl.getResource(resourceLocation);
      if (resource == null) {
        LOGGER.debug("Couldn't find XML schema [" + systemId + "]: " + resourceLocation);
        return null;
      }
      // The caller expects the stream in the InputSource to be open, so this cannot be closed before returning.
      try {
        InputStream is = getInputStreamWithCacheControl(resource);

        InputSource source = new InputSource(is);
        source.setPublicId(publicId);
        source.setSystemId(systemId);
        if (LOGGER.isDebugEnabled()) {
          LOGGER.debug("Found XML schema [" + systemId + "] in classpath: " + resourceLocation);
        }
        return source;
      } catch (IOException e) {
        LOGGER.warn("Error resolving entity [" + systemId + "]: " + resourceLocation, e);
      }
    }
    return null;
  }

  /**
   * Load the specified schema mappings.
   */
  private Map<String, String> getMuleSchemaMappings(SchemaMappingsUtils schemaMappingsUtils) {
    Map<String, String> schemaMappings = schemaMappingsUtils.getMuleSchemasMappings();
    Map<String, String> springMappings = schemaMappingsUtils.getSpringSchemasMappings();
    schemaMappings.putAll(springMappings);
    return schemaMappings;
  }

  private ClassLoader getClassLoaderToUse(ClassLoader classLoader) {
    ClassLoader classLoaderToUse = classLoader;
    if (classLoaderToUse == null) {
      classLoaderToUse = getDefaultClassLoader();
    }

    return classLoaderToUse;
  }

  private ClassLoader getDefaultClassLoader() {
    ClassLoader cl = null;
    try {
      cl = Thread.currentThread().getContextClassLoader();
    } catch (Throwable ex) {
      // Cannot access thread context ClassLoader - falling back...
    }
    if (cl == null) {
      // No thread context class loader -> use class loader of this class.
      cl = muleImplementationsLoader;
      if (cl == null) {
        // getClassLoader() returning null indicates the bootstrap ClassLoader
        try {
          cl = ClassLoader.getSystemClassLoader();
        } catch (Throwable ex) {
          // Cannot access system ClassLoader - oh well, maybe the caller can live with null...
        }
      }
    }
    return cl;
  }

  /**
   * Load the specified schema mappings.
   */
  private Map<String, String> getAppPluginsSchemaMappings(SchemaMappingsUtils schemaMappingsUtils) {
    return schemaMappingsUtils.getSchemaMappings(CUSTOM_SCHEMA_MAPPINGS_LOCATION,
                                                 () -> getClassLoaderToUse(this.deploymentClassLoader));
  }

}
