/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal.xml;

import static org.mule.runtime.api.util.collection.Collectors.toImmutableList;
import static org.mule.runtime.dsl.api.xml.XmlNamespaceInfoProvider.loadXmlNamespaceInfoProviders;

import static java.lang.Thread.currentThread;
import static java.util.stream.Collectors.toList;

import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.dsl.api.xml.XmlNamespaceInfo;
import org.mule.runtime.dsl.api.xml.XmlNamespaceInfoProvider;

import java.util.List;
import java.util.Set;

import com.google.common.collect.ImmutableList;

/**
 * This code was moved from mule project
 */
public class XmlNamespaceInfoProviderSupplier {


  /**
   * Creates an list of {@link XmlNamespaceInfoProvider} based on the list of {@link ExtensionModel}s used by the artifact.
   * <p/>
   * The list of {@link XmlNamespaceInfoProvider} will be discovered based on those extensions and the one discovered using by
   * SPI.
   *
   * @param extensionModels the {@link ExtensionModel}s of the artifact that contains the configuration.
   * @return a list of {@link XmlNamespaceInfoProvider}.
   */
  public static List<XmlNamespaceInfoProvider> createFromExtensionModels(Set<ExtensionModel> extensionModels) {
    return ImmutableList.<XmlNamespaceInfoProvider>builder()
        .add(createStaticNamespaceInfoProviders(extensionModels))
        .addAll(discoverRuntimeXmlNamespaceInfoProvider())
        .addAll(discoverArtifactNamespaceInfoProvider())
        .build();
  }

  private static List<XmlNamespaceInfoProvider> discoverArtifactNamespaceInfoProvider() {
    return loadXmlNamespaceInfoProviders(currentThread().getContextClassLoader()).collect(toList());
  }

  private static List<XmlNamespaceInfoProvider> discoverRuntimeXmlNamespaceInfoProvider() {
    return loadXmlNamespaceInfoProviders().collect(toList());
  }

  private static XmlNamespaceInfoProvider createStaticNamespaceInfoProviders(Set<ExtensionModel> extensionModels) {
    List<XmlNamespaceInfo> extensionNamespaces = extensionModels.stream()
        .map(ext -> new StaticXmlNamespaceInfo(ext.getXmlDslModel().getNamespace(), ext.getXmlDslModel().getPrefix()))
        .collect(toImmutableList());

    return new StaticXmlNamespaceInfoProvider(extensionNamespaces);
  }


}
