/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.test.internal.xml.metadata;

import static org.mule.runtime.ast.api.xml.AstXmlParser.builder;
import static org.mule.runtime.ast.api.xml.AstXmlParserAttribute.CLOSING_TAG_END_COLUMN;
import static org.mule.runtime.ast.api.xml.AstXmlParserAttribute.CLOSING_TAG_END_LINE;
import static org.mule.runtime.ast.api.xml.AstXmlParserAttribute.CLOSING_TAG_START_COLUMN;
import static org.mule.runtime.ast.api.xml.AstXmlParserAttribute.CLOSING_TAG_START_LINE;
import static org.mule.runtime.ast.api.xml.AstXmlParserAttribute.IS_SELF_CLOSING;
import static org.mule.runtime.ast.api.xml.AstXmlParserAttribute.OPENING_TAG_END_COLUMN;
import static org.mule.runtime.ast.api.xml.AstXmlParserAttribute.OPENING_TAG_END_LINE;
import static org.mule.runtime.ast.api.xml.AstXmlParserAttribute.OPENING_TAG_START_COLUMN;
import static org.mule.runtime.ast.api.xml.AstXmlParserAttribute.OPENING_TAG_START_LINE;
import static org.mule.runtime.ast.test.AllureConstants.ArtifactAst.ARTIFACT_AST;
import static org.mule.runtime.ast.test.AllureConstants.ArtifactAst.Metadata.SOURCE_LOCATION;

import static java.util.Optional.of;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.not;
import static org.hamcrest.Matchers.nullValue;
import static org.hamcrest.core.Is.is;

import org.mule.runtime.ast.api.ArtifactAst;
import org.mule.runtime.ast.api.ComponentAst;
import org.mule.runtime.ast.api.ComponentMetadataAst;
import org.mule.runtime.ast.api.xml.AstXmlParser;
import org.mule.runtime.ast.test.internal.xml.DefaultAstXmlParserTestCase;
import org.mule.runtime.dsl.api.ConfigResource;

import java.net.URISyntaxException;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;

import org.junit.Before;
import org.junit.Test;

import io.qameta.allure.Feature;
import io.qameta.allure.Issue;
import io.qameta.allure.Story;

@Feature(ARTIFACT_AST)
@Story(SOURCE_LOCATION)
public class LocationMetadataTestCase {

  private ClassLoader classLoader;
  private AstXmlParser parser;

  @Before
  public void before() {
    classLoader = DefaultAstXmlParserTestCase.class.getClassLoader();

    parser = builder()
        .withSchemaValidationsDisabled()
        .withPropertyResolver(propertyKey -> propertyKey)
        .build();
  }

  @Test
  @Issue("MULE-19634")
  public void propertyLocationMetadata() throws URISyntaxException {
    final String resourceName = "metadata/flow-with-source-and-operations.xml";
    final URL resource = this.getClass().getResource("/" + resourceName);
    final ArtifactAst simpleAst = parser.parse(resource);

    ComponentAst globalProperty = simpleAst.topLevelComponents().get(0);

    ComponentMetadataAst metadata = globalProperty.getMetadata();
    assertThat(metadata.getFileName().get(), is(resourceName));
    assertThat(metadata.getFileUri().get(), is(resource.toURI()));
    assertThat(metadata.getStartLine().getAsInt(), is(11));
    assertThat(metadata.getStartLine().getAsInt(), is(11));
    assertThat(metadata.getStartColumn().getAsInt(), is(5));
    assertThat(metadata.getEndLine().getAsInt(), is(11));
    assertThat(metadata.getEndColumn().getAsInt(), is(49));

    assertThat(OPENING_TAG_START_LINE.get(metadata), is(of(11)));
    assertThat(OPENING_TAG_START_COLUMN.get(metadata), is(of(5)));
    assertThat(OPENING_TAG_END_LINE.get(metadata), is(of(11)));
    assertThat(OPENING_TAG_END_COLUMN.get(metadata), is(of(49)));
    assertThat(CLOSING_TAG_START_LINE.get(metadata), is(of(11)));
    assertThat(CLOSING_TAG_START_COLUMN.get(metadata), is(of(5)));
    assertThat(CLOSING_TAG_END_LINE.get(metadata), is(of(11)));
    assertThat(CLOSING_TAG_END_COLUMN.get(metadata), is(of(49)));
    assertThat(IS_SELF_CLOSING.get(metadata), is(of(true)));
  }

  @Test
  public void configLocationMetadata() throws URISyntaxException {
    final String resourceName = "metadata/flow-with-source-and-operations.xml";
    final URL resource = this.getClass().getResource("/" + resourceName);
    final ArtifactAst simpleAst = parser.parse(resource);

    ComponentAst httpConfig = simpleAst.topLevelComponents().get(1);

    ComponentMetadataAst metadata = httpConfig.getMetadata();
    assertThat(metadata.getFileName().get(), is(resourceName));
    assertThat(metadata.getFileUri().get(), is(resource.toURI()));
    assertThat(metadata.getStartLine().getAsInt(), is(12));
    assertThat(metadata.getStartColumn().getAsInt(), is(5));
    assertThat(metadata.getEndLine().getAsInt(), is(14));
    assertThat(metadata.getEndColumn().getAsInt(), is(28));

    assertThat(OPENING_TAG_START_LINE.get(metadata), is(of(12)));
    assertThat(OPENING_TAG_START_COLUMN.get(metadata), is(of(5)));
    assertThat(OPENING_TAG_END_LINE.get(metadata), is(of(12)));
    assertThat(OPENING_TAG_END_COLUMN.get(metadata), is(of(46)));
    assertThat(CLOSING_TAG_START_LINE.get(metadata), is(of(14)));
    assertThat(CLOSING_TAG_START_COLUMN.get(metadata), is(of(5)));
    assertThat(CLOSING_TAG_END_LINE.get(metadata), is(of(14)));
    assertThat(CLOSING_TAG_END_COLUMN.get(metadata), is(of(28)));
    assertThat(IS_SELF_CLOSING.get(metadata).orElse(false), is(false));
  }

  @Test
  public void flowLocationMetadata() throws URISyntaxException {
    final String resourceName = "metadata/flow-with-source-and-operations.xml";
    final URL resource = this.getClass().getResource("/" + resourceName);
    final ArtifactAst simpleAst = parser.parse(resource);

    ComponentAst flow = simpleAst.topLevelComponents().get(2);

    ComponentMetadataAst metadata = flow.getMetadata();
    assertThat(metadata.getFileName().get(), is(resourceName));
    assertThat(metadata.getFileUri().get(), is(resource.toURI()));
    assertThat(metadata.getStartLine().getAsInt(), is(16));
    assertThat(metadata.getStartColumn().getAsInt(), is(5));
    assertThat(metadata.getEndLine().getAsInt(), is(38));
    assertThat(metadata.getEndColumn().getAsInt(), is(12));

    assertThat(OPENING_TAG_START_LINE.get(metadata), is(of(16)));
    assertThat(OPENING_TAG_START_COLUMN.get(metadata), is(of(5)));
    assertThat(OPENING_TAG_END_LINE.get(metadata), is(of(16)));
    assertThat(OPENING_TAG_END_COLUMN.get(metadata), is(of(29)));
    assertThat(CLOSING_TAG_START_LINE.get(metadata), is(of(38)));
    assertThat(CLOSING_TAG_START_COLUMN.get(metadata), is(of(5)));
    assertThat(CLOSING_TAG_END_LINE.get(metadata), is(of(38)));
    assertThat(CLOSING_TAG_END_COLUMN.get(metadata), is(of(12)));
    assertThat(IS_SELF_CLOSING.get(metadata).orElse(false), is(false));
  }

  @Test
  @Issue("MULE-19634")
  public void sourceLocationMetadata() throws URISyntaxException {
    final String resourceName = "metadata/flow-with-source-and-operations.xml";
    final URL resource = this.getClass().getResource("/" + resourceName);
    final ArtifactAst simpleAst = parser.parse(resource);

    ComponentAst flow = simpleAst.topLevelComponents().get(2);
    ComponentAst source = flow.directChildren().get(0);

    ComponentMetadataAst metadata = source.getMetadata();
    assertThat(metadata.getFileName().get(), is(resourceName));
    assertThat(metadata.getFileUri().get(), is(resource.toURI()));
    assertThat(metadata.getStartLine().getAsInt(), is(17));
    assertThat(metadata.getStartColumn().getAsInt(), is(9));
    assertThat(metadata.getEndLine().getAsInt(), is(17));
    assertThat(metadata.getEndColumn().getAsInt(), is(63));

    assertThat(OPENING_TAG_START_LINE.get(metadata), is(of(17)));
    assertThat(OPENING_TAG_START_COLUMN.get(metadata), is(of(9)));
    assertThat(OPENING_TAG_END_LINE.get(metadata), is(of(17)));
    assertThat(OPENING_TAG_END_COLUMN.get(metadata), is(of(63)));
    assertThat(CLOSING_TAG_START_LINE.get(metadata), is(of(17)));
    assertThat(CLOSING_TAG_START_COLUMN.get(metadata), is(of(9)));
    assertThat(CLOSING_TAG_END_LINE.get(metadata), is(of(17)));
    assertThat(CLOSING_TAG_END_COLUMN.get(metadata), is(of(63)));
    assertThat(IS_SELF_CLOSING.get(metadata), is(of(true)));
  }

  @Test
  public void operationTransformLocationMetadata() throws URISyntaxException {
    final String resourceName = "metadata/flow-with-source-and-operations.xml";
    final URL resource = this.getClass().getResource("/" + resourceName);
    final ArtifactAst simpleAst = parser.parse(resource);

    ComponentAst flow = simpleAst.topLevelComponents().get(2);
    ComponentAst transform = flow.directChildren().get(1);

    ComponentMetadataAst metadata = transform.getMetadata();
    assertThat(metadata.getFileName().get(), is(resourceName));
    assertThat(metadata.getFileUri().get(), is(resource.toURI()));
    assertThat(metadata.getStartLine().getAsInt(), is(18));
    assertThat(metadata.getStartColumn().getAsInt(), is(9));
    assertThat(metadata.getEndLine().getAsInt(), is(27));
    assertThat(metadata.getEndColumn().getAsInt(), is(24));

    assertThat(OPENING_TAG_START_LINE.get(metadata), is(of(18)));
    assertThat(OPENING_TAG_START_COLUMN.get(metadata), is(of(9)));
    assertThat(OPENING_TAG_END_LINE.get(metadata), is(of(18)));
    assertThat(OPENING_TAG_END_COLUMN.get(metadata), is(of(48)));
    assertThat(CLOSING_TAG_START_LINE.get(metadata), is(of(27)));
    assertThat(CLOSING_TAG_START_COLUMN.get(metadata), is(of(9)));
    assertThat(CLOSING_TAG_END_LINE.get(metadata), is(of(27)));
    assertThat(CLOSING_TAG_END_COLUMN.get(metadata), is(of(24)));
    assertThat(IS_SELF_CLOSING.get(metadata).orElse(false), is(false));
  }

  @Test
  public void operationLoggerLocationMetadata() throws URISyntaxException {
    final String resourceName = "metadata/flow-with-source-and-operations.xml";
    final URL resource = this.getClass().getResource("/" + resourceName);
    final ArtifactAst simpleAst = parser.parse(resource);

    ComponentAst flow = simpleAst.topLevelComponents().get(2);
    ComponentAst logger = flow.directChildren().get(2);

    ComponentMetadataAst metadata = logger.getMetadata();
    assertThat(metadata.getFileName().get(), is(resourceName));
    assertThat(metadata.getFileUri().get(), is(resource.toURI()));
    assertThat(metadata.getStartLine().getAsInt(), is(28));
    assertThat(metadata.getStartColumn().getAsInt(), is(9));
    assertThat(metadata.getEndLine().getAsInt(), is(28));
    assertThat(metadata.getEndColumn().getAsInt(), is(63));

    assertThat(OPENING_TAG_START_LINE.get(metadata), is(of(28)));
    assertThat(OPENING_TAG_START_COLUMN.get(metadata), is(of(9)));
    assertThat(OPENING_TAG_END_LINE.get(metadata), is(of(28)));
    assertThat(OPENING_TAG_END_COLUMN.get(metadata), is(of(63)));
    assertThat(CLOSING_TAG_START_LINE.get(metadata), is(of(28)));
    assertThat(CLOSING_TAG_START_COLUMN.get(metadata), is(of(9)));
    assertThat(CLOSING_TAG_END_LINE.get(metadata), is(of(28)));
    assertThat(CLOSING_TAG_END_COLUMN.get(metadata), is(of(63)));
    assertThat(IS_SELF_CLOSING.get(metadata), is(of(true)));
  }

  @Test
  public void choiceRouterLocationMetadata() throws URISyntaxException {
    final String resourceName = "metadata/flow-with-source-and-operations.xml";
    final URL resource = this.getClass().getResource("/" + resourceName);
    final ArtifactAst simpleAst = parser.parse(resource);

    ComponentAst flow = simpleAst.topLevelComponents().get(2);
    ComponentAst choice = flow.directChildren().get(4);

    ComponentMetadataAst metadata = choice.getMetadata();
    assertThat(metadata.getFileName().get(), is(resourceName));
    assertThat(metadata.getFileUri().get(), is(resource.toURI()));
    assertThat(metadata.getStartLine().getAsInt(), is(30));
    assertThat(metadata.getStartColumn().getAsInt(), is(9));
    assertThat(metadata.getEndLine().getAsInt(), is(37));
    assertThat(metadata.getEndColumn().getAsInt(), is(18));

    assertThat(OPENING_TAG_START_LINE.get(metadata), is(of(30)));
    assertThat(OPENING_TAG_START_COLUMN.get(metadata), is(of(9)));
    assertThat(OPENING_TAG_END_LINE.get(metadata), is(of(30)));
    assertThat(OPENING_TAG_END_COLUMN.get(metadata), is(of(17)));
    assertThat(CLOSING_TAG_START_LINE.get(metadata), is(of(37)));
    assertThat(CLOSING_TAG_START_COLUMN.get(metadata), is(of(9)));
    assertThat(CLOSING_TAG_END_LINE.get(metadata), is(of(37)));
    assertThat(CLOSING_TAG_END_COLUMN.get(metadata), is(of(18)));
    assertThat(IS_SELF_CLOSING.get(metadata).orElse(false), is(false));
  }

  @Test
  public void operationRouteLocationMetadata() throws URISyntaxException {
    final String resourceName = "metadata/flow-with-source-and-operations.xml";
    final URL resource = this.getClass().getResource("/" + resourceName);
    final ArtifactAst simpleAst = parser.parse(resource);

    ComponentAst flow = simpleAst.topLevelComponents().get(2);
    ComponentAst choice = flow.directChildren().get(4);
    ComponentAst firstRoute = choice.directChildren().get(0);

    ComponentMetadataAst metadata = firstRoute.getMetadata();
    assertThat(metadata.getFileName().get(), is(resourceName));
    assertThat(metadata.getFileUri().get(), is(resource.toURI()));
    assertThat(metadata.getStartLine().getAsInt(), is(31));
    assertThat(metadata.getStartColumn().getAsInt(), is(13));
    assertThat(metadata.getEndLine().getAsInt(), is(33));
    assertThat(metadata.getEndColumn().getAsInt(), is(20));

    assertThat(OPENING_TAG_START_LINE.get(metadata), is(of(31)));
    assertThat(OPENING_TAG_START_COLUMN.get(metadata), is(of(13)));
    assertThat(OPENING_TAG_END_LINE.get(metadata), is(of(31)));
    assertThat(OPENING_TAG_END_COLUMN.get(metadata), is(of(49)));
    assertThat(CLOSING_TAG_START_LINE.get(metadata), is(of(33)));
    assertThat(CLOSING_TAG_START_COLUMN.get(metadata), is(of(13)));
    assertThat(CLOSING_TAG_END_LINE.get(metadata), is(of(33)));
    assertThat(CLOSING_TAG_END_COLUMN.get(metadata), is(of(20)));
    assertThat(IS_SELF_CLOSING.get(metadata).orElse(false), is(false));
  }


  @Test
  public void operationInChoiceRouteLocationMetadata() throws URISyntaxException {
    final String resourceName = "metadata/flow-with-source-and-operations.xml";
    final URL resource = this.getClass().getResource("/" + resourceName);
    final ArtifactAst simpleAst = parser.parse(resource);

    ComponentAst flow = simpleAst.topLevelComponents().get(2);
    ComponentAst choice = flow.directChildren().get(4);
    ComponentAst firstRoute = choice.directChildren().get(0);
    ComponentAst componentInChoiceRoute = firstRoute.directChildren().get(0);

    ComponentMetadataAst metadata = componentInChoiceRoute.getMetadata();
    assertThat(metadata.getFileName().get(), is(resourceName));
    assertThat(metadata.getFileUri().get(), is(resource.toURI()));
    assertThat(metadata.getStartLine().getAsInt(), is(32));
    assertThat(metadata.getStartColumn().getAsInt(), is(17));
    assertThat(metadata.getEndLine().getAsInt(), is(32));
    assertThat(metadata.getEndColumn().getAsInt(), is(71));

    assertThat(OPENING_TAG_START_LINE.get(metadata), is(of(32)));
    assertThat(OPENING_TAG_START_COLUMN.get(metadata), is(of(17)));
    assertThat(OPENING_TAG_END_LINE.get(metadata), is(of(32)));
    assertThat(OPENING_TAG_END_COLUMN.get(metadata), is(of(71)));
    assertThat(CLOSING_TAG_START_LINE.get(metadata), is(of(32)));
    assertThat(CLOSING_TAG_START_COLUMN.get(metadata), is(of(17)));
    assertThat(CLOSING_TAG_END_LINE.get(metadata), is(of(32)));
    assertThat(CLOSING_TAG_END_COLUMN.get(metadata), is(of(71)));
    assertThat(IS_SELF_CLOSING.get(metadata), is(of(true)));
  }

  @Test
  public void fileNameAndUriAreSameWhenLoadedFromConfigResource() {
    final String resourceName = "metadata/flow-with-source-and-operations.xml";
    final URL resource = this.getClass().getResource("/" + resourceName);

    assertThat(resource, is(not(nullValue())));

    final ConfigResource configResource = new ConfigResource(resource);
    final ArtifactAst simpleAstFromUrl = parser.parse(resource);
    final ArtifactAst simpleAstFromConfigResource = parser.parse(configResource);

    final List<ComponentAst> componentAstsFromUrl = new ArrayList<>();
    componentAstsFromUrl.add(simpleAstFromUrl.topLevelComponents().get(2));
    componentAstsFromUrl.add(componentAstsFromUrl.get(0).directChildren().get(4));
    componentAstsFromUrl.add(componentAstsFromUrl.get(1).directChildren().get(0));
    componentAstsFromUrl.add(componentAstsFromUrl.get(2).directChildren().get(0));

    final List<ComponentAst> componentAstsFromConfigResource = new ArrayList<>();
    componentAstsFromConfigResource.add(simpleAstFromConfigResource.topLevelComponents().get(2));
    componentAstsFromConfigResource.add(componentAstsFromConfigResource.get(0).directChildren().get(4));
    componentAstsFromConfigResource.add(componentAstsFromConfigResource.get(1).directChildren().get(0));
    componentAstsFromConfigResource.add(componentAstsFromConfigResource.get(2).directChildren().get(0));

    for (int i = 0; i < componentAstsFromUrl.size(); i++) {
      final ComponentMetadataAst metadataFromUrl = componentAstsFromUrl.get(i).getMetadata();
      final ComponentMetadataAst metadataFromConfigResource = componentAstsFromConfigResource.get(i).getMetadata();
      assertThat(metadataFromConfigResource.getFileName(), is(equalTo(metadataFromUrl.getFileName())));
      assertThat(metadataFromConfigResource.getFileUri(), is(equalTo(metadataFromUrl.getFileUri())));
    }
  }

  @Test
  public void locationMetadataIsCorrectWhenOpeningTagsAreMultiline() throws URISyntaxException {
    final String resourceName = "metadata/multiline-opening-tags.xml";
    final URL resource = this.getClass().getResource("/" + resourceName);
    final ArtifactAst simpleAst = parser.parse(resource);

    ComponentAst httpConfig = simpleAst.topLevelComponents().get(1);

    ComponentMetadataAst metadata = httpConfig.getMetadata();
    assertThat(metadata.getStartLine().getAsInt(), is(10));
    assertThat(metadata.getStartColumn().getAsInt(), is(5));
    assertThat(metadata.getEndLine().getAsInt(), is(13));
    assertThat(metadata.getEndColumn().getAsInt(), is(28));

    assertThat(OPENING_TAG_START_LINE.get(metadata), is(of(10)));
    assertThat(OPENING_TAG_START_COLUMN.get(metadata), is(of(5)));
    assertThat(OPENING_TAG_END_LINE.get(metadata), is(of(11)));
    assertThat(OPENING_TAG_END_COLUMN.get(metadata), is(of(32)));
    assertThat(CLOSING_TAG_START_LINE.get(metadata), is(of(13)));
    assertThat(CLOSING_TAG_START_COLUMN.get(metadata), is(of(5)));
    assertThat(CLOSING_TAG_END_LINE.get(metadata), is(of(13)));
    assertThat(CLOSING_TAG_END_COLUMN.get(metadata), is(of(28)));
    assertThat(IS_SELF_CLOSING.get(metadata).orElse(false), is(false));
  }

  @Test
  public void locationMetadataIsCorrectWhenSelfClosingTagsAreMultiline() throws URISyntaxException {
    final String resourceName = "metadata/multiline-opening-tags.xml";
    final URL resource = this.getClass().getResource("/" + resourceName);
    final ArtifactAst simpleAst = parser.parse(resource);

    ComponentAst globalProperty = simpleAst.topLevelComponents().get(0);

    ComponentMetadataAst metadata = globalProperty.getMetadata();
    assertThat(metadata.getStartLine().getAsInt(), is(8));
    assertThat(metadata.getStartColumn().getAsInt(), is(5));
    assertThat(metadata.getEndLine().getAsInt(), is(9));
    assertThat(metadata.getEndColumn().getAsInt(), is(37));

    assertThat(OPENING_TAG_START_LINE.get(metadata), is(of(8)));
    assertThat(OPENING_TAG_START_COLUMN.get(metadata), is(of(5)));
    assertThat(OPENING_TAG_END_LINE.get(metadata), is(of(9)));
    assertThat(OPENING_TAG_END_COLUMN.get(metadata), is(of(37)));
    assertThat(CLOSING_TAG_START_LINE.get(metadata), is(of(8)));
    assertThat(CLOSING_TAG_START_COLUMN.get(metadata), is(of(5)));
    assertThat(CLOSING_TAG_END_LINE.get(metadata), is(of(9)));
    assertThat(CLOSING_TAG_END_COLUMN.get(metadata), is(of(37)));
    assertThat(IS_SELF_CLOSING.get(metadata), is(of(true)));
  }

}
