/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.test.internal.xml;

import static org.mule.runtime.ast.test.AllureConstants.ArtifactAst.ARTIFACT_AST;
import static org.mule.runtime.ast.test.internal.xml.AllureXmlParserConstants.DslParsing.CACHE;
import static org.mule.runtime.dsl.api.xml.parser.XmlConfigurationDocumentLoader.noValidationDocumentLoader;

import static java.util.Collections.emptySet;
import static java.util.Collections.singletonList;
import static java.util.Optional.empty;

import static org.mockito.ArgumentMatchers.any;
import static org.mockito.ArgumentMatchers.anySet;
import static org.mockito.ArgumentMatchers.anyString;
import static org.mockito.Mockito.mockStatic;
import static org.mockito.Mockito.never;

import org.mule.runtime.api.util.Pair;
import org.mule.runtime.api.util.xmlsecurity.XMLSecureFactories;
import org.mule.runtime.ast.api.ArtifactType;
import org.mule.runtime.ast.api.builder.ArtifactAstBuilder;
import org.mule.runtime.ast.internal.xml.DefaultAstXmlParser;
import org.mule.runtime.dsl.api.ConfigResource;

import org.w3c.dom.Document;
import org.xml.sax.helpers.DefaultHandler;

import org.junit.Test;

import org.mockito.MockedStatic;

import io.qameta.allure.Feature;
import io.qameta.allure.Story;

@Feature(ARTIFACT_AST)
@Story(CACHE)
public class AstXmlParserCacheTestCase {

  @Test
  public void byDocumentCache() {
    DefaultAstXmlParser parser = new DefaultAstXmlParser(ArtifactType.APPLICATION, emptySet(), null, empty(), p -> p, null);

    Document document = noValidationDocumentLoader().loadDocument(() -> XMLSecureFactories.createDefault().getSAXParserFactory(),
                                                                  new DefaultHandler(),
                                                                  "simple.xml",
                                                                  this.getClass().getClassLoader()
                                                                      .getResourceAsStream("simple.xml"),
                                                                  null);

    parser.parseDocument("testApp",
                         singletonList(new Pair<>("simple.xml",
                                                  document)));

    try (MockedStatic<ArtifactAstBuilder> astBuilder = mockStatic(ArtifactAstBuilder.class)) {
      parser.parseDocument("testApp",
                           singletonList(new Pair<>("simple.xml",
                                                    document)));

      // no new ast was built for the cached config file
      astBuilder.verify(() -> ArtifactAstBuilder.builder(anyString(),
                                                         any(),
                                                         anySet(),
                                                         any(),
                                                         any()),
                        never());
    }
  }

  @Test
  public void byResourceCache() throws Exception {
    DefaultAstXmlParser parser = new DefaultAstXmlParser(ArtifactType.APPLICATION, emptySet(), null, empty(), p -> p, null);

    ConfigResource simpleResource = new ConfigResource("simple.xml", this.getClass().getClassLoader().getResource("simple.xml"));

    parser.parse("testApp", simpleResource);

    try (MockedStatic<ArtifactAstBuilder> astBuilder = mockStatic(ArtifactAstBuilder.class)) {
      parser.parse("testApp", simpleResource);

      // no new ast was built for the cached config file
      astBuilder.verify(() -> ArtifactAstBuilder.builder(anyString(),
                                                         any(),
                                                         anySet(),
                                                         any(),
                                                         any()),
                        never());
    }
  }
}
