/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal.xml.resolver;

import static org.mule.runtime.dsl.internal.util.SchemaMappingsUtils.resolveSystemId;

import static java.lang.String.format;
import static java.lang.Thread.currentThread;
import static java.nio.charset.StandardCharsets.UTF_8;

import org.mule.runtime.api.dsl.DslResolvingContext;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.ast.api.model.ExtensionModelHelper;
import org.mule.runtime.extension.api.dsl.syntax.resources.spi.ExtensionSchemaGenerator;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Set;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.xml.sax.EntityResolver;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

/**
 * Custom implementation of resolver for schemas where it will delegate to our custom resolver, then if not found will try to
 * generate the XSDs from the extensions (through {@link }).
 * <p>
 * This code was moved from {@code mule} project
 *
 * @since 1.0
 */
public class ModuleDelegatingEntityResolver implements EntityResolver {

  private static final Logger LOGGER = LoggerFactory.getLogger(ModuleDelegatingEntityResolver.class);

  private final Set<ExtensionModel> extensions;
  private final EntityResolver muleEntityResolver;
  private final ExtensionSchemaGenerator extensionSchemaFactory;
  private final DslResolvingContext dslResolvingContext;
  private final ExtensionModelHelper extensionModelHelper;
  private final ResolveEntityFailStrategy failStrategy;

  /**
   * Returns an instance of {@link ModuleDelegatingEntityResolver}
   *
   * @param extensions             fallback set to dynamically generate schemas from {@link ExtensionModel} if the current
   *                               {@link #muleEntityResolver} delegates return null when resolving the entity.
   * @param extensionSchemaFactory
   * @param dslResolvingContext
   */
  public ModuleDelegatingEntityResolver(Set<ExtensionModel> extensions, ExtensionSchemaGenerator extensionSchemaFactory,
                                        DslResolvingContext dslResolvingContext, ExtensionModelHelper extensionModelHelper,
                                        ResolveEntityFailStrategy failStrategy) {
    ClassLoader classLoader = currentThread().getContextClassLoader();
    this.muleEntityResolver = new MuleCustomEntityResolver(classLoader);
    this.extensions = extensions;

    this.extensionSchemaFactory = extensionSchemaFactory;
    this.dslResolvingContext = dslResolvingContext;
    this.extensionModelHelper = extensionModelHelper;
    this.failStrategy = failStrategy;
  }

  @Override
  public InputSource resolveEntity(String publicId, String systemId) throws SAXException, IOException {
    if (LOGGER.isDebugEnabled()) {
      LOGGER.debug(format("Looking schema for public identifier(publicId): '%s', system identifier(systemId): '%s'",
                          publicId == null ? "" : publicId,
                          systemId));
    }

    systemId = resolveSystemId(systemId);

    InputSource inputSource;
    inputSource = muleEntityResolver.resolveEntity(publicId, systemId);
    if (inputSource == null) {
      inputSource = generateFromExtensions(publicId, systemId);
    }
    if (inputSource == null) {
      failStrategy.execute(publicId, systemId);
    }
    return inputSource;
  }

  private InputSource generateFromExtensions(String publicId, String systemId) {
    return extensions.stream()
        .filter(em -> systemId.equals(em.getXmlDslModel().getSchemaLocation()))
        .map(em -> {
          InputStream schema = getSchemaFromExtension(em);
          InputSource inputSource = new InputSource(schema);
          inputSource.setPublicId(publicId);
          inputSource.setSystemId(systemId);

          return inputSource;
        })
        .findAny()
        .orElse(null);
  }

  /**
   * Given an {@link ExtensionModel} it will generate the XSD for it.
   *
   * @param extensionModel extension to generate the schema for
   * @return the bytes that represent the schema for the {@code extensionModel}
   */
  private InputStream getSchemaFromExtension(ExtensionModel extensionModel) {
    return new ByteArrayInputStream(extensionSchemaFactory
        .generate(extensionModel, dslResolvingContext, extensionModelHelper.getDslSyntaxResolverFor(extensionModel))
        .getBytes(UTF_8));
  }
}
