/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.test.internal.serialization.dto;

import static org.mule.metadata.api.model.MetadataFormat.JAVA;
import static org.mule.runtime.api.component.ComponentIdentifier.builder;
import static org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType.UNKNOWN;
import static org.mule.runtime.api.meta.model.parameter.ParameterGroupModel.DEFAULT_GROUP_NAME;
import static org.mule.runtime.api.meta.model.parameter.ParameterRole.BEHAVIOUR;
import static org.mule.runtime.api.meta.model.parameter.ParameterRole.CONTENT;
import static org.mule.runtime.ast.AllureConstants.ArtifactAstSerialization.AST_SERIALIZATION;
import static org.mule.runtime.ast.AllureConstants.ArtifactAstSerialization.AST_SERIALIZATION_ENRICH;
import static org.mule.runtime.ast.api.ArtifactType.APPLICATION;
import static org.mule.runtime.ast.internal.serialization.dto.ComponentAstDTOModelType.CONSTRUCT;
import static org.mule.runtime.ast.internal.serialization.dto.ComponentAstDTOModelType.INFRASTRUCTURE;
import static org.mule.runtime.ast.internal.serialization.dto.ComponentAstDTOModelType.OPERATION;
import static org.mule.runtime.ast.internal.serialization.dto.ComponentAstDTOModelType.TYPE;

import static java.util.Arrays.asList;
import static java.util.Collections.emptyList;
import static java.util.Collections.emptySet;
import static java.util.Collections.singleton;
import static java.util.Collections.singletonList;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static java.util.Optional.ofNullable;
import static java.util.function.UnaryOperator.identity;
import static java.util.stream.Collectors.toList;

import static org.hamcrest.MatcherAssert.assertThat;

import static org.mockito.Mockito.doAnswer;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.never;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

import static org.hamcrest.Matchers.containsInAnyOrder;
import static org.hamcrest.Matchers.everyItem;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.not;
import static org.hamcrest.Matchers.notNullValue;
import static org.hamcrest.Matchers.nullValue;
import static org.hamcrest.collection.IsCollectionWithSize.hasSize;
import static org.mockito.ArgumentMatchers.any;

import org.mule.metadata.api.annotation.ExampleAnnotation;
import org.mule.metadata.api.annotation.TypeAnnotation;
import org.mule.metadata.api.builder.BaseTypeBuilder;
import org.mule.metadata.api.builder.ObjectTypeBuilder;
import org.mule.metadata.api.model.MetadataFormat;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.api.model.ObjectType;
import org.mule.metadata.api.model.UnionType;
import org.mule.metadata.api.visitor.MetadataTypeVisitor;
import org.mule.runtime.api.component.ComponentIdentifier;
import org.mule.runtime.api.component.location.ComponentLocation;
import org.mule.runtime.api.message.ErrorType;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.meta.model.ParameterDslConfiguration;
import org.mule.runtime.api.meta.model.config.ConfigurationModel;
import org.mule.runtime.api.meta.model.construct.ConstructModel;
import org.mule.runtime.api.meta.model.operation.OperationModel;
import org.mule.runtime.api.meta.model.parameter.ParameterGroupModel;
import org.mule.runtime.api.meta.model.parameter.ParameterModel;
import org.mule.runtime.api.meta.model.parameter.ParameterRole;
import org.mule.runtime.api.meta.model.parameter.ParameterizedModel;
import org.mule.runtime.api.meta.type.TypeCatalog;
import org.mule.runtime.ast.api.ComponentAst;
import org.mule.runtime.ast.api.ComponentParameterAst;
import org.mule.runtime.ast.api.serialization.ExtensionModelResolver;
import org.mule.runtime.ast.internal.builder.PropertiesResolver;
import org.mule.runtime.ast.internal.model.DefaultExtensionModelHelper;
import org.mule.runtime.ast.internal.model.ParameterModelUtils;
import org.mule.runtime.ast.internal.serialization.dto.ArtifactAstDTO;
import org.mule.runtime.ast.internal.serialization.dto.ComponentAstDTO;
import org.mule.runtime.ast.internal.serialization.dto.ComponentAstDTOModelType;
import org.mule.runtime.ast.internal.serialization.dto.ComponentGenerationInformationDTO;
import org.mule.runtime.ast.internal.serialization.dto.ComponentMetadataAstDTO;
import org.mule.runtime.ast.internal.serialization.dto.ComponentParameterAstDTO;
import org.mule.runtime.ast.internal.serialization.dto.ErrorTypeRepositoryDTO;
import org.mule.runtime.ast.internal.serialization.dto.ExtensionModelDTO;
import org.mule.runtime.ast.internal.serialization.dto.ParameterValueContainer;
import org.mule.runtime.ast.internal.serialization.resolver.DefaultGenerationInformationResolver;
import org.mule.runtime.ast.internal.serialization.resolver.GenerationInformationResolver;
import org.mule.runtime.ast.internal.serialization.visitor.UnionTypesVisitor;
import org.mule.runtime.ast.testobjects.DummyExtensionModel;
import org.mule.runtime.extension.api.dsl.syntax.DslElementSyntax;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.function.UnaryOperator;

import org.junit.Before;
import org.junit.Test;

import io.qameta.allure.Feature;
import io.qameta.allure.Issue;
import io.qameta.allure.Story;

@Feature(AST_SERIALIZATION)
@Story(AST_SERIALIZATION_ENRICH)
public class ArtifactAstDTOTestCase {

  private static final String CONSTRUCT_MODEL_NAME = "constructModelName";
  private static final String CONSTRUCT_MODEL_ELEMENT_NAME = "construct-model-name";
  private static final String CONSTRUCT_PARAM_GROUP_MODEL_NAME = "constructParamGroupModelName";
  private static final String CONSTRUCT_STRING_LIST_PARAM_MODEL_NAME = "stringListParamModelName";
  private static final String CONSTRUCT_COMPONENT_LIST_PARAM_MODEL_NAME = "componentListParamModelName";
  private static final String CONSTRUCT_SIMPLE_PARAM_MODEL_NAME = "simpleParamModelName";
  private static final String CONSTRUCT_COMPONENT_AST_PARAM_MODEL_NAME = "componentAstParamModelName";

  private static final String DUMMY_NAMESPACE = "dummyNamespace";
  private static final String DUMMY_NAMESPACE_URI = "dummyNamespaceUri";

  private static final String OPERATION_MODEL_NAME = "operationModelName";
  private static final String OPERATION_MODEL_ELEMENT_NAME = "operation-model-name";
  private static final String OPERATION_PARAM_GROUP_MODEL_NAME = "operationParamGroupModelName";
  private static final String OPERATION_SIMPLE_PARAM_MODEL_NAME = "simpleParamModelName";

  private static final String INFRASTRUCTURE_TYPE_IDENTIFIER = "error-mapping";
  private static final String INFRASTRUCTURE_PARAMETER_IDENTIFIER = "error-mapping";
  private static final String INFRASTRUCTURE_PARAMETER_PARAMETERIZED_MODEL_NAME = "ErrorMapping";
  private static final String INFRASTRUCTURE_TYPE_PARAMETER_MODEL_GROUP_NAME = "ErrorMapping";
  private static final String INFRASTRUCTURE_PARAMETER_PARAMETER_GROUP_MODEL_NAME = "ErrorMapping";
  private static final String INFRASTRUCTURE_TYPE_PARAMETERIZED_MODEL_NAME = "infrastructureTypeParameterizedModel";
  private static final String INFRASTRUCTURE_TYPE_PARAMETER_MODEL_NAME = "name";
  private static final String INFRASTRUCTURE_PARAMETER_PARAMETER_MODEL = "name";

  private ExtensionModelResolver extensionModelResolver;
  private GenerationInformationResolver generationInformationResolver;
  private OperationModel operationModel;
  private ConstructModel constructModel;
  private ConstructModel paramValueConstructModel;
  private ExtensionModelDTO muleExtensionModelDTO;
  private ExtensionModelDTO dummyExtensionModelDTO;
  private ArtifactAstDTO artifactAstDTOWithOneComponent;
  private ArtifactAstDTO artifactAstDTOWithStringListParam;
  private ArtifactAstDTO artifactAstDTOWithComponentListParam;
  private ArtifactAstDTO artifactAstDTOWithOneInfrastructureType;
  private ArtifactAstDTO artifactAstDTOWithOneInfrastructureParameter;

  @Before
  public void setUp() {
    this.muleExtensionModelDTO = new ExtensionModelDTO("mule");
    this.dummyExtensionModelDTO = new ExtensionModelDTO(DUMMY_NAMESPACE);

    // This ArtifactAstDTO has 3 to level components. The first one is a construct with 2 parameters and no direct children. The
    // first of this parameter's value is actually a ComponentAstDTO and its model indicates it is an operation, while the second
    // one has an empty value. The operation in turn has one parameter with an empty value.
    // The second top level component is an infrastructure type.
    // The third top level component is an infrastructure parameter.
    this.artifactAstDTOWithOneComponent = makeArtifactAstDTOWithOneComponent();
    this.artifactAstDTOWithStringListParam = makeArtifactAstDTOWithStringListParam();
    this.artifactAstDTOWithComponentListParam = makeArtifactAstDTOWithComponentListParam();
    this.artifactAstDTOWithOneInfrastructureType = makeArtifactAstDTOWithOneInfrastructureType();
    this.artifactAstDTOWithOneInfrastructureParameter = makeArtifactAstDTOWithOneInfrastructureParameter();
    this.extensionModelResolver = this.initExtensionModelResolver();

    this.generationInformationResolver = new DefaultGenerationInformationResolver();
  }

  @Test
  public void testEnrichCallsExtensionModelResolver_WhenEnrichingArtifactAstDTO() {
    // Given
    ComponentGenerationInformationDTO componentGenerationInformationDTO = mock(ComponentGenerationInformationDTO.class);
    DslElementSyntax dslElementSyntax = mock(DslElementSyntax.class);
    when(componentGenerationInformationDTO.getSyntax()).thenReturn(of(dslElementSyntax));
    when(dslElementSyntax.getChild(any())).thenReturn(of(dslElementSyntax));
    when(dslElementSyntax.isWrapped()).thenReturn(false);

    // When
    artifactAstDTOWithOneComponent.enrich(extensionModelResolver, generationInformationResolver, new ParameterModelUtils());

    // Then
    verify(extensionModelResolver, times(3)).resolve(DUMMY_NAMESPACE);
    verify(extensionModelResolver, times(1)).resolve("mule");
  }

  @Test
  public void testEnrichResolves2DependencyDTOsIntoDependencies_WhenEnrichingArtifactAstDTOWith2DependencyDTOsAnd3ComponentsTotal() {
    // Given

    // When
    artifactAstDTOWithOneComponent.enrich(extensionModelResolver, generationInformationResolver, new ParameterModelUtils());

    // Then
    verify(extensionModelResolver, times(1)).resolve(muleExtensionModelDTO.getName());
    // One for dependency + one for each component
    verify(extensionModelResolver, times(3)).resolve(dummyExtensionModelDTO.getName());

    assertThat(artifactAstDTOWithOneComponent.dependencies().stream().map(ExtensionModel::getName).collect(toList()),
               containsInAnyOrder(muleExtensionModelDTO.getName(), dummyExtensionModelDTO.getName()));
  }

  @Test
  public void testEnrichResolves2DependencyDTOsIntoDependencies_WhenEnrichingArtifactAstDTOWith2DependencyDTOs() {
    // Given

    // When
    artifactAstDTOWithOneInfrastructureType.enrich(extensionModelResolver, generationInformationResolver,
                                                   new ParameterModelUtils());

    // Then
    verify(extensionModelResolver, times(2)).resolve(muleExtensionModelDTO.getName());
    verify(extensionModelResolver, times(1)).resolve(dummyExtensionModelDTO.getName());

    assertThat(artifactAstDTOWithOneInfrastructureType.dependencies().stream().map(ExtensionModel::getName).collect(toList()),
               containsInAnyOrder(muleExtensionModelDTO.getName(), dummyExtensionModelDTO.getName()));
  }

  @Test
  public void testEnrichResolvesTopLevelComponentAstModel_WhenEnrichingArtifactAstDTOWithATopLevelComponentAst() {
    // Given

    // When
    artifactAstDTOWithOneComponent.enrich(extensionModelResolver, generationInformationResolver, new ParameterModelUtils());

    // Then
    ComponentAst complexComponent = artifactAstDTOWithOneComponent.topLevelComponents().get(0);

    // I have resolved the model of the first top level component
    assertThat(complexComponent.getModel(Object.class).isPresent(), is(true));
    assertThat(complexComponent.getModel(ParameterizedModel.class).get().getName(), is(CONSTRUCT_MODEL_NAME));
  }

  @Test
  public void testEnrichResolvesComponentParameterAstModels_WhenEnrichingArtifactAstDTOWithATopLevelComponentWithAParameter() {
    // Given

    // When
    artifactAstDTOWithOneComponent.enrich(extensionModelResolver, generationInformationResolver, new ParameterModelUtils());

    // Then
    ComponentAst complexComponent = artifactAstDTOWithOneComponent.topLevelComponents().get(0);
    List<ComponentParameterAst> firstTopLevelComponentParameters = new ArrayList<>(complexComponent.getParameters());

    // I have resolved the models of the first parameter
    ComponentParameterAst firstConstructComponentParameterAst = firstTopLevelComponentParameters.get(0);
    assertThat(firstConstructComponentParameterAst.getModel(), notNullValue());
    assertThat(firstConstructComponentParameterAst.getModel().getName(), is(CONSTRUCT_COMPONENT_AST_PARAM_MODEL_NAME));
    assertThat(firstConstructComponentParameterAst.getGroupModel(), notNullValue());
    assertThat(firstConstructComponentParameterAst.getGroupModel().getName(), is(CONSTRUCT_PARAM_GROUP_MODEL_NAME));

    // I have resolved the models of the second parameter of the construct
    assertThat(firstTopLevelComponentParameters.get(1).getModel(), notNullValue());
    assertThat(firstTopLevelComponentParameters.get(1).getModel().getName(), is(CONSTRUCT_SIMPLE_PARAM_MODEL_NAME));
    assertThat(firstTopLevelComponentParameters.get(1).getGroupModel(), notNullValue());
    assertThat(firstTopLevelComponentParameters.get(1).getGroupModel().getName(), is(CONSTRUCT_PARAM_GROUP_MODEL_NAME));
  }

  @Test
  public void testEnrichResolvesModelOfComponentAstThatIsValueOfParameter_WhenEnrichingArtifactAstDTOWithTopLevelComponentThatHasParameterWhoseValueIsAComponentAst() {
    // Given

    // When
    artifactAstDTOWithOneComponent.enrich(extensionModelResolver, generationInformationResolver, new ParameterModelUtils());

    // Then
    ComponentAst complexComponent = artifactAstDTOWithOneComponent.topLevelComponents().get(0);
    List<ComponentParameterAst> firstTopLevelComponentParameters = new ArrayList<>(complexComponent.getParameters());

    ComponentParameterAst firstConstructComponentParameterAst = firstTopLevelComponentParameters.get(0);

    // The value of the first parameter is set as configured
    assertThat(firstConstructComponentParameterAst.getValue().isRight(), is(true));
    assertThat(firstConstructComponentParameterAst.getValue().getRight(), notNullValue());
    ComponentAst valueOfTheFirstParameterOfTheConstruct =
        (ComponentAst) (firstConstructComponentParameterAst.getValue().getRight());
    assertThat(valueOfTheFirstParameterOfTheConstruct, notNullValue());

    // I have resolved the model of the ComponentAst, representing an operation, that is the value of the first parameter
    assertThat(valueOfTheFirstParameterOfTheConstruct.getModel(Object.class).isPresent(), is(true));
    assertThat(valueOfTheFirstParameterOfTheConstruct.getModel(ParameterizedModel.class).get().getName(),
               is(OPERATION_MODEL_NAME));
  }

  @Test
  public void testEnrichResolvesModelOfComponentParameterAstOfComponentAstThatIsValueOfAnotherParameter_WhenEnrichingArtifactAstDTOWithTopLevelComponentThatHasParameterWhoseValueIsAComponentAstThatHasParameters() {
    // Given

    // When
    artifactAstDTOWithOneComponent.enrich(extensionModelResolver, generationInformationResolver, new ParameterModelUtils());

    // Then
    ComponentAst complexComponent = artifactAstDTOWithOneComponent.topLevelComponents().get(0);
    List<ComponentParameterAst> firstTopLevelComponentParameters = new ArrayList<>(complexComponent.getParameters());

    ComponentParameterAst firstConstructComponentParameterAst = firstTopLevelComponentParameters.get(0);
    ComponentAst valueOfTheFirstParameterOfTheConstruct =
        (ComponentAst) (firstConstructComponentParameterAst.getValue().getRight());

    ComponentParameterAst valueComponentParameters =
        new ArrayList<>(valueOfTheFirstParameterOfTheConstruct.getParameters()).get(0);

    // I have resolved the models of the only operation parameter
    assertThat(valueComponentParameters.getModel(), notNullValue());
    assertThat(valueComponentParameters.getModel().getName(), is(OPERATION_SIMPLE_PARAM_MODEL_NAME));
    assertThat(valueComponentParameters.getGroupModel(), notNullValue());
    assertThat(valueComponentParameters.getGroupModel().getName(), is(OPERATION_PARAM_GROUP_MODEL_NAME));
  }

  @Test
  @Issue("W-15677894")
  public void testEnrichResolvesModelOfComponentParameterAstOfComponentAstThatIsValueOfAnotherParameter_WhenEnrichingArtifactAstDTOWithTopLevelComponentThatHasParameterWhoseValueIsAListOfSimpleValues() {
    // When
    artifactAstDTOWithStringListParam.enrich(extensionModelResolver, generationInformationResolver, new ParameterModelUtils());

    // Then
    ComponentAst complexComponent = artifactAstDTOWithStringListParam.topLevelComponents().get(0);
    List<ComponentParameterAst> firstTopLevelComponentParameters = new ArrayList<>(complexComponent.getParameters());

    ComponentParameterAst firstConstructComponentParameterAst = firstTopLevelComponentParameters.get(0);
    List<ComponentAst> valueOfTheFirstParameterOfTheConstruct =
        (List<ComponentAst>) (firstConstructComponentParameterAst.getValue().getRight());

    ComponentParameterAst valueComponentParameters =
        new ArrayList<>(valueOfTheFirstParameterOfTheConstruct.get(0).getParameters()).get(0);

    // I have resolved the models of the only operation parameter
    assertThat(valueComponentParameters.getModel(), notNullValue());
    assertThat(valueComponentParameters.getModel().getName(), is("value"));
    assertThat(valueComponentParameters.getGroupModel(), notNullValue());
    assertThat(valueComponentParameters.getGroupModel().getName(), is(DEFAULT_GROUP_NAME));
  }

  @Test
  @Issue("W-15677894")
  public void testEnrichResolvesModelOfComponentParameterAstOfComponentAstThatIsValueOfAnotherParameter_WhenEnrichingArtifactAstDTOWithTopLevelComponentThatHasParameterWhoseValueIsAListOfComponents() {
    // When
    artifactAstDTOWithComponentListParam.enrich(extensionModelResolver, generationInformationResolver, new ParameterModelUtils());

    // Then
    ComponentAst complexComponent = artifactAstDTOWithComponentListParam.topLevelComponents().get(0);
    List<ComponentParameterAst> firstTopLevelComponentParameters = new ArrayList<>(complexComponent.getParameters());

    ComponentParameterAst firstConstructComponentParameterAst = firstTopLevelComponentParameters.get(0);
    List<ComponentAst> valueOfTheFirstParameterOfTheConstruct =
        (List<ComponentAst>) (firstConstructComponentParameterAst.getValue().getRight());

    ComponentParameterAst valueComponentParameters =
        new ArrayList<>(valueOfTheFirstParameterOfTheConstruct.get(0).getParameters()).get(0);

    // I have resolved the models of the only operation parameter
    assertThat(valueComponentParameters.getModel(), notNullValue());
    assertThat(valueComponentParameters.getModel().getName(), is(CONSTRUCT_COMPONENT_AST_PARAM_MODEL_NAME));
    assertThat(valueComponentParameters.getGroupModel(), notNullValue());
    assertThat(valueComponentParameters.getGroupModel().getName(), is(CONSTRUCT_PARAM_GROUP_MODEL_NAME));
  }

  @Test
  public void testEnrichResolvesInfrastructureParameterModel_WhenEnrichingArtifactAstDTOWithATopLevelInfrastructureComponentAst() {
    // Given

    // When
    artifactAstDTOWithOneInfrastructureParameter.enrich(extensionModelResolver, generationInformationResolver,
                                                        new ParameterModelUtils());

    // Then
    // I have resolved the models of the third top level component which is an infrastructure parameter
    ComponentAst infrastructureParameterComponent = artifactAstDTOWithOneInfrastructureParameter.topLevelComponents().get(0);
    assertThat(infrastructureParameterComponent.getModel(ParameterizedModel.class).isPresent(), is(true));
    assertThat(infrastructureParameterComponent.getModel(ParameterizedModel.class).get().getName(),
               is(INFRASTRUCTURE_PARAMETER_PARAMETERIZED_MODEL_NAME));
  }

  @Test
  public void testUpdatePropertiesResolverCallsPropertyResolverSetMappingFunction_WhenUpdatingPropertyResolversMappingFunction() {
    // Given
    PropertiesResolver propertiesResolver = mock(PropertiesResolver.class);
    UnaryOperator<String> newMappingFunction = identity();
    artifactAstDTOWithOneComponent.setPropertiesResolver(propertiesResolver);

    // When
    artifactAstDTOWithOneComponent.updatePropertiesResolver(newMappingFunction);

    // Then
    verify(propertiesResolver, times(1)).setMappingFunction(newMappingFunction);
  }

  @Test
  @Issue("MULE-19806")
  public void testResolveParameterModelsParameterGroupModelsAndGenerationInformationResolvesUnionType_WhenResolvingModelsForParameterWithUnionType() {
    // Given
    DefaultGenerationInformationResolver generationInformationResolver = mock(DefaultGenerationInformationResolver.class);

    ParameterModel parameterModel = mock(ParameterModel.class);
    UnionType unionType = mock(UnionType.class);
    when(parameterModel.getType()).thenReturn(unionType);
    when(parameterModel.getName()).thenReturn("parameterModelName");

    ParameterGroupModel parameterGroupModel = mock(ParameterGroupModel.class);
    when(parameterGroupModel.getName()).thenReturn("parameterGroupModelName");

    ComponentAstDTO componentParameterValue = mock(ComponentAstDTO.class);
    ComponentIdentifier componentIdentifier =
        builder().name("aName").namespace("aNamespace").namespaceUri("aNamespaceUri").build();
    when(componentParameterValue.getIdentifier()).thenReturn(componentIdentifier);
    when(componentParameterValue.getModelType()).thenReturn(TYPE);
    when(componentParameterValue.getModelName()).thenReturn("someTypeId");
    when(componentParameterValue.getExtensionModelDTO()).thenReturn(mock(ExtensionModelDTO.class));

    ComponentParameterAstDTO componentParameterAstDTO =
        makeComponentParameterAstDTO(new ParameterValueContainer(null, componentParameterValue));

    ExtensionModel extensionModel = mock(ExtensionModel.class);
    when(this.extensionModelResolver.resolve("someExtensionModel")).thenReturn(extensionModel);

    ConfigurationModel componentAstDTOModel = mock(ConfigurationModel.class);
    when(componentAstDTOModel.getAllParameterModels()).thenReturn(singletonList(parameterModel));
    when(componentAstDTOModel.getParameterGroupModels()).thenReturn(singletonList(parameterGroupModel));

    when(extensionModel.getConfigurationModel("configurationModelName")).thenReturn(of(componentAstDTOModel));

    ComponentAstDTO componentAstDTO =
        makeComponentAstDTO(new ExtensionModelDTO("someExtensionModel"), null, builder()
            .name("configuration")
            .namespace("dummyNamespace")
            .namespaceUri("dummyNamespaceUri")
            .build(), "configurationModelName",
                            ComponentAstDTOModelType.CONFIGURATION, singletonList(componentParameterAstDTO), new ArrayList<>());

    componentAstDTO.setParameterizedModel(componentAstDTOModel);

    DefaultExtensionModelHelper extensionModelHelper = mock(DefaultExtensionModelHelper.class);
    ComponentGenerationInformationDTO componentGenerationInformationDTO = mock(ComponentGenerationInformationDTO.class);
    DslElementSyntax componentSyntax = mock(DslElementSyntax.class);
    when(componentGenerationInformationDTO.getSyntax()).thenReturn(ofNullable(componentSyntax));
    DslElementSyntax childSyntax = mock(DslElementSyntax.class);
    when(componentSyntax.getChild(any())).thenReturn(of(childSyntax));
    when(childSyntax.isWrapped()).thenReturn(false);
    when(generationInformationResolver.resolveComponentAstGenerationInformation(null, componentAstDTO, extensionModelHelper))
        .thenReturn(componentGenerationInformationDTO);

    MetadataType metadataType = mock(MetadataType.class);
    when(unionType.getTypes()).thenReturn(singletonList(metadataType));

    doAnswer(invocation -> {
      UnionTypesVisitor argument = invocation.getArgument(0);
      argument.visitUnion(unionType);
      return argument;
    }).when(unionType).accept(any(UnionTypesVisitor.class));

    DslElementSyntax dslElementSyntax = mock(DslElementSyntax.class);
    when(dslElementSyntax.getElementName()).thenReturn("aName");

    TypeCatalog typeCatalog = mock(TypeCatalog.class);
    when(typeCatalog.getType("someTypeId")).thenReturn(of(mock(ObjectType.class)));
    when(extensionModelHelper.getTypeCatalog()).thenReturn(typeCatalog);

    // When
    componentAstDTO.resolveModelsRecursively(null, extensionModelHelper, extensionModelResolver, generationInformationResolver,
                                             new ParameterModelUtils());

    // Then
    verify(unionType, times(1)).accept(any(UnionTypesVisitor.class));
    verify(generationInformationResolver, never())
        .resolveComponentParameterAstGenerationInformation(componentParameterAstDTO, componentAstDTO, extensionModelHelper);
  }

  @Test
  @Issue("W-12244895")
  public void enrichExtensionModelNotFound() {
    ExtensionModelDTO nonExistantExtensionModelDTO = new ExtensionModelDTO("nonExistant");
    Set<ExtensionModelDTO> extensionModelDTOS =
        new HashSet<>(asList(muleExtensionModelDTO, dummyExtensionModelDTO, nonExistantExtensionModelDTO));

    ArtifactAstDTO artifactAstDTO =
        new ArtifactAstDTO("myApp", APPLICATION, extensionModelDTOS, emptyList(), mock(ErrorTypeRepositoryDTO.class),
                           emptySet());


    artifactAstDTO.enrich(extensionModelResolver, generationInformationResolver, new ParameterModelUtils());

    verify(extensionModelResolver).resolve("nonExistant");
    assertThat(artifactAstDTO.dependencies(), hasSize(2));
    assertThat(artifactAstDTO.dependencies(), everyItem(is(not(nullValue()))));
  }

  private ComponentParameterAstDTO makeComponentParameterAstDTO(ParameterValueContainer value) {
    return new ComponentParameterAstDTO(value, null, false, null, "parameterGroupModelName", "parameterModelName");
  }

  private ArtifactAstDTO makeArtifactAstDTOWithOneComponent() {
    operationModel = this.initOperationModel();
    constructModel = this.initConstructModel();

    Set<ExtensionModelDTO> extensionModelDTOS = new HashSet<>(asList(muleExtensionModelDTO, dummyExtensionModelDTO));

    ComponentAstDTO constructWithAChildOperationComponentAstDTO =
        makeComponentAstDTO(dummyExtensionModelDTO, "constructWithAChildOperation", builder()
            .name(CONSTRUCT_MODEL_ELEMENT_NAME).namespace(DUMMY_NAMESPACE).namespaceUri(DUMMY_NAMESPACE_URI).build(),
                            CONSTRUCT_MODEL_NAME, CONSTRUCT,
                            asList(makeParameterWithAComponentAstValue(),
                                   makeParameterWithAnEmptyValue(CONSTRUCT_PARAM_GROUP_MODEL_NAME,
                                                                 CONSTRUCT_SIMPLE_PARAM_MODEL_NAME)),
                            emptyList());

    List<ComponentAstDTO> topLevelComponents = singletonList(constructWithAChildOperationComponentAstDTO);

    // I don't really need to enrich this
    ErrorTypeRepositoryDTO errorTypeRepositoryDTO = makeErrorTypeRepository();

    return new ArtifactAstDTO("myApp", APPLICATION, extensionModelDTOS, topLevelComponents, errorTypeRepositoryDTO,
                              emptySet());
  }

  private ArtifactAstDTO makeArtifactAstDTOWithStringListParam() {
    operationModel = this.initOperationModel();
    constructModel = this.initConstructModel();

    Set<ExtensionModelDTO> extensionModelDTOS = new HashSet<>(asList(muleExtensionModelDTO, dummyExtensionModelDTO));

    ComponentAstDTO constructWithAChildOperationComponentAstDTO =
        makeComponentAstDTO(dummyExtensionModelDTO, "constructWithAChildOperation", builder()
            .name(CONSTRUCT_MODEL_ELEMENT_NAME).namespace(DUMMY_NAMESPACE).namespaceUri(DUMMY_NAMESPACE_URI).build(),
                            CONSTRUCT_MODEL_NAME, CONSTRUCT,
                            asList(makeParameterWithAStringListValue(CONSTRUCT_PARAM_GROUP_MODEL_NAME,
                                                                     CONSTRUCT_STRING_LIST_PARAM_MODEL_NAME)),
                            emptyList());

    List<ComponentAstDTO> topLevelComponents = singletonList(constructWithAChildOperationComponentAstDTO);

    // I don't really need to enrich this
    ErrorTypeRepositoryDTO errorTypeRepositoryDTO = makeErrorTypeRepository();

    return new ArtifactAstDTO("myApp", APPLICATION, extensionModelDTOS, topLevelComponents, errorTypeRepositoryDTO,
                              emptySet());
  }

  private ArtifactAstDTO makeArtifactAstDTOWithComponentListParam() {
    operationModel = this.initOperationModel();
    constructModel = this.initConstructModel();
    paramValueConstructModel = this.initParamValueConstructModel();

    Set<ExtensionModelDTO> extensionModelDTOS = new HashSet<>(asList(muleExtensionModelDTO, dummyExtensionModelDTO));

    ComponentAstDTO constructWithAChildOperationComponentAstDTO =
        makeComponentAstDTO(dummyExtensionModelDTO, "constructWithAChildOperation", builder()
            .name(CONSTRUCT_MODEL_ELEMENT_NAME).namespace(DUMMY_NAMESPACE).namespaceUri(DUMMY_NAMESPACE_URI).build(),
                            CONSTRUCT_MODEL_NAME, CONSTRUCT,
                            asList(makeParameterWithAComponentListValue(CONSTRUCT_PARAM_GROUP_MODEL_NAME,
                                                                        CONSTRUCT_COMPONENT_LIST_PARAM_MODEL_NAME)),
                            emptyList());

    List<ComponentAstDTO> topLevelComponents = singletonList(constructWithAChildOperationComponentAstDTO);

    // I don't really need to enrich this
    ErrorTypeRepositoryDTO errorTypeRepositoryDTO = makeErrorTypeRepository();

    return new ArtifactAstDTO("myApp", APPLICATION, extensionModelDTOS, topLevelComponents, errorTypeRepositoryDTO,
                              emptySet());
  }

  private ArtifactAstDTO makeArtifactAstDTOWithOneInfrastructureParameter() {


    Set<ExtensionModelDTO> dependencies = new HashSet<>(asList(muleExtensionModelDTO, dummyExtensionModelDTO));

    ComponentAstDTO infrastructureParameterComponentAstDTO =
        makeComponentAstDTO(dummyExtensionModelDTO, "infrastructureParameter",
                            builder().name(INFRASTRUCTURE_PARAMETER_IDENTIFIER).namespace("mule")
                                .namespaceUri("http://www.mulesoft.org/schema/mule/core").build(),
                            INFRASTRUCTURE_PARAMETER_PARAMETERIZED_MODEL_NAME, INFRASTRUCTURE,
                            singletonList(makeParameterWithAStringValue(INFRASTRUCTURE_PARAMETER_PARAMETER_GROUP_MODEL_NAME,
                                                                        INFRASTRUCTURE_PARAMETER_PARAMETER_MODEL)),
                            emptyList());

    List<ComponentAstDTO> topLevelComponents = singletonList(infrastructureParameterComponentAstDTO);

    return new ArtifactAstDTO("myApp", APPLICATION, dependencies, topLevelComponents, mock(ErrorTypeRepositoryDTO.class),
                              emptySet());
  }

  private ArtifactAstDTO makeArtifactAstDTOWithOneInfrastructureType() {
    Set<ExtensionModelDTO> dependencies = new HashSet<>(asList(muleExtensionModelDTO, dummyExtensionModelDTO));

    ComponentAstDTO infrastructureTypeComponentAstDTO =
        makeComponentAstDTO(muleExtensionModelDTO, "errorMappings",
                            builder().name(INFRASTRUCTURE_TYPE_IDENTIFIER).namespace("mule")
                                .namespaceUri("http://www.mulesoft.org/schema/mule/core").build(),
                            INFRASTRUCTURE_TYPE_PARAMETERIZED_MODEL_NAME, INFRASTRUCTURE,
                            singletonList(makeParameterWithAStringValue(INFRASTRUCTURE_TYPE_PARAMETER_MODEL_GROUP_NAME,
                                                                        INFRASTRUCTURE_TYPE_PARAMETER_MODEL_NAME)),
                            emptyList());

    List<ComponentAstDTO> topLevelComponents = singletonList(infrastructureTypeComponentAstDTO);

    return new ArtifactAstDTO("myApp", APPLICATION, dependencies, topLevelComponents, mock(ErrorTypeRepositoryDTO.class),
                              emptySet());
  }

  private ErrorTypeRepositoryDTO makeErrorTypeRepository() {

    return new ErrorTypeRepositoryDTO(makeErrorType("ANY", null), makeErrorType("SOURCE", makeErrorType("ANY", null)),
                                      makeErrorType("SOURCE_RESPONSE", makeErrorType("ANY", null)),
                                      makeErrorType("CRITICAL", makeErrorType("ANY", null)),
                                      new HashSet<>(asList(makeErrorType("FIRST", makeErrorType("ANY", null)),
                                                           makeErrorType("SECOND", makeErrorType("ANY", null)))),
                                      new HashSet<>(asList(makeErrorType("FIRST_INTERNAL", makeErrorType("ANY", null)),
                                                           makeErrorType("SECOND_INTERNAL", makeErrorType("ANY", null)),
                                                           makeErrorType("THIRD_INTERNAL", makeErrorType("ANY", null)))));
  }

  private ErrorType makeErrorType(String name, ErrorType parentErrorType) {
    ErrorType anyErrorType = mock(ErrorType.class);
    when(anyErrorType.getIdentifier()).thenReturn(name);
    when(anyErrorType.getNamespace()).thenReturn(DUMMY_NAMESPACE);
    when(anyErrorType.getParentErrorType()).thenReturn(parentErrorType);
    return anyErrorType;
  }

  private OperationModel initOperationModel() {
    OperationModel operationModel = mock(OperationModel.class);
    when(operationModel.getName()).thenReturn(OPERATION_MODEL_NAME);
    ParameterGroupModel mockParameterGroupModel = mockParameterGroupModel(OPERATION_PARAM_GROUP_MODEL_NAME);
    when(operationModel.getParameterGroupModels()).thenReturn(singletonList(mockParameterGroupModel));
    ParameterModel mockParameterModel =
        mockParameterModel(OPERATION_SIMPLE_PARAM_MODEL_NAME, CONTENT, new TestMetadataType());
    when(operationModel.getAllParameterModels())
        .thenReturn(singletonList(mockParameterModel));
    return operationModel;
  }

  private ConstructModel initConstructModel() {
    ConstructModel constructModel = mock(ConstructModel.class);
    when(constructModel.getName()).thenReturn(CONSTRUCT_MODEL_NAME);
    ParameterGroupModel parameterGroupModel = mockParameterGroupModel(CONSTRUCT_PARAM_GROUP_MODEL_NAME);
    when(constructModel.getParameterGroupModels())
        .thenReturn(singletonList(parameterGroupModel));
    ParameterModel mockParameterWithComponentValueModel =
        mockParameterModel(CONSTRUCT_COMPONENT_AST_PARAM_MODEL_NAME, BEHAVIOUR, new TestMetadataType());
    ParameterModel mockParameterWithStringListModel =
        mockParameterModel(CONSTRUCT_STRING_LIST_PARAM_MODEL_NAME, BEHAVIOUR,
                           new BaseTypeBuilder(JAVA).arrayType().of(new BaseTypeBuilder(JAVA).stringType()).build());

    final ObjectTypeBuilder objectTypeBuilder = new BaseTypeBuilder(JAVA).objectType();
    objectTypeBuilder.addField()
        .key("fieldA")
        .value(new BaseTypeBuilder(JAVA).stringType())
        .build();
    objectTypeBuilder.addField()
        .key("fieldB")
        .value(new BaseTypeBuilder(JAVA).stringType())
        .build();
    MetadataType componentType = objectTypeBuilder.build();
    ParameterModel mockParameterWithComponentListModel =
        mockParameterModel(CONSTRUCT_COMPONENT_LIST_PARAM_MODEL_NAME, BEHAVIOUR,
                           new BaseTypeBuilder(JAVA).arrayType().of(componentType).build());
    ParameterModel mockParameterModel =
        mockParameterModel(CONSTRUCT_SIMPLE_PARAM_MODEL_NAME, CONTENT, new TestMetadataType());
    when(constructModel.getAllParameterModels())
        .thenReturn(asList(mockParameterWithComponentValueModel, mockParameterWithStringListModel,
                           mockParameterWithComponentListModel, mockParameterModel));
    return constructModel;
  }

  private ConstructModel initParamValueConstructModel() {
    ConstructModel constructModel = mock(ConstructModel.class);
    when(constructModel.getName()).thenReturn("componentsList");
    ParameterGroupModel parameterGroupModel = mockParameterGroupModel(CONSTRUCT_PARAM_GROUP_MODEL_NAME);
    when(constructModel.getParameterGroupModels())
        .thenReturn(singletonList(parameterGroupModel));
    ParameterModel mockParameterWithComponentValueModel =
        mockParameterModel(CONSTRUCT_COMPONENT_AST_PARAM_MODEL_NAME, BEHAVIOUR, new TestMetadataType());
    ParameterModel mockParameterWithStringListModel =
        mockParameterModel(CONSTRUCT_STRING_LIST_PARAM_MODEL_NAME, BEHAVIOUR,
                           new BaseTypeBuilder(JAVA).arrayType().of(new BaseTypeBuilder(JAVA).stringType()).build());
    ParameterModel mockParameterModel =
        mockParameterModel(CONSTRUCT_SIMPLE_PARAM_MODEL_NAME, CONTENT, new TestMetadataType());
    when(constructModel.getAllParameterModels())
        .thenReturn(asList(mockParameterWithComponentValueModel, mockParameterWithStringListModel,
                           mockParameterModel));
    return constructModel;
  }

  private ExtensionModelResolver initExtensionModelResolver() {
    ExtensionModelResolver extensionModelResolver = mock(ExtensionModelResolver.class);

    when(extensionModelResolver.resolve(muleExtensionModelDTO.getName()))
        .thenReturn(new DummyExtensionModel(muleExtensionModelDTO.getName()));

    when(extensionModelResolver.resolve(dummyExtensionModelDTO.getName()))
        .thenReturn(new DummyExtensionModel(dummyExtensionModelDTO.getName(), singletonList(operationModel), emptyList(),
                                            emptyList(), asList(constructModel, paramValueConstructModel), new ArrayList<>()));
    return extensionModelResolver;
  }

  private ComponentParameterAstDTO makeParameterWithAnEmptyValue(String parameterGroupModelName, String parameterModelName) {
    return new ComponentParameterAstDTO(null, "unresolvedValue", false, new ComponentMetadataAstDTO(), parameterGroupModelName,
                                        parameterModelName);
  }

  private ComponentParameterAstDTO makeParameterWithAStringValue(String parameterGroupModelName, String parameterModelName) {
    ParameterValueContainer value = new ParameterValueContainer(null, "some value");
    return new ComponentParameterAstDTO(value, "unresolvedValue", false, makeEmptyComponentMetadataAstDTO(),
                                        parameterGroupModelName, parameterModelName);
  }

  private ComponentParameterAstDTO makeParameterWithAStringListValue(String parameterGroupModelName, String parameterModelName) {
    ParameterValueContainer value = new ParameterValueContainer(null,
                                                                asList(makeComponentAstDTO(dummyExtensionModelDTO, "stringsList",
                                                                                           builder()
                                                                                               .name("strings-list")
                                                                                               .namespace(DUMMY_NAMESPACE)
                                                                                               .namespaceUri(DUMMY_NAMESPACE_URI)
                                                                                               .build(),
                                                                                           "stringsList", CONSTRUCT,
                                                                                           asList(makeParameterWithAStringValue(DEFAULT_GROUP_NAME,
                                                                                                                                "value")),
                                                                                           emptyList()),
                                                                       makeComponentAstDTO(dummyExtensionModelDTO, "stringsList",
                                                                                           builder()
                                                                                               .name("strings-list")
                                                                                               .namespace(DUMMY_NAMESPACE)
                                                                                               .namespaceUri(DUMMY_NAMESPACE_URI)
                                                                                               .build(),
                                                                                           "stringsList", CONSTRUCT,
                                                                                           asList(makeParameterWithAStringValue(DEFAULT_GROUP_NAME,
                                                                                                                                "value")),
                                                                                           emptyList())));

    return new ComponentParameterAstDTO(value, "unresolvedValue", false, makeEmptyComponentMetadataAstDTO(),
                                        parameterGroupModelName, parameterModelName);
  }

  private ComponentParameterAstDTO makeParameterWithAComponentListValue(String parameterGroupModelName,
                                                                        String parameterModelName) {
    ParameterValueContainer value = new ParameterValueContainer(null,
                                                                asList(makeComponentAstDTO(dummyExtensionModelDTO,
                                                                                           "componentsList",
                                                                                           builder()
                                                                                               .name("components-list")
                                                                                               .namespace(DUMMY_NAMESPACE)
                                                                                               .namespaceUri(DUMMY_NAMESPACE_URI)
                                                                                               .build(),
                                                                                           "componentsList", CONSTRUCT,
                                                                                           asList(makeParameterWithAComponentAstValue()),
                                                                                           emptyList()),
                                                                       makeComponentAstDTO(dummyExtensionModelDTO,
                                                                                           "componentsList",
                                                                                           builder()
                                                                                               .name("components-list")
                                                                                               .namespace(DUMMY_NAMESPACE)
                                                                                               .namespaceUri(DUMMY_NAMESPACE_URI)
                                                                                               .build(),
                                                                                           "componentsList", CONSTRUCT,
                                                                                           asList(makeParameterWithAComponentAstValue()),
                                                                                           emptyList())));

    return new ComponentParameterAstDTO(value, "unresolvedValue", false, makeEmptyComponentMetadataAstDTO(),
                                        parameterGroupModelName, parameterModelName);
  }

  private ComponentParameterAstDTO makeParameterWithAComponentAstValue() {
    return new ComponentParameterAstDTO(new ParameterValueContainer(null,
                                                                    makeComponentAstDTO(dummyExtensionModelDTO,
                                                                                        "level1children1Id",
                                                                                        builder()
                                                                                            .name(OPERATION_MODEL_ELEMENT_NAME)
                                                                                            .namespace("dummyNamespace")
                                                                                            .namespaceUri("dummyNamespaceUri")
                                                                                            .build(),
                                                                                        OPERATION_MODEL_NAME, OPERATION,
                                                                                        singletonList(makeParameterWithAnEmptyValue(OPERATION_PARAM_GROUP_MODEL_NAME,
                                                                                                                                    OPERATION_SIMPLE_PARAM_MODEL_NAME)),
                                                                                        emptyList())),
                                        "unresolvedValue", false, makeEmptyComponentMetadataAstDTO(),
                                        CONSTRUCT_PARAM_GROUP_MODEL_NAME,
                                        CONSTRUCT_COMPONENT_AST_PARAM_MODEL_NAME);
  }

  private ComponentAstDTO makeComponentAstDTO(ExtensionModelDTO dummyExtensionModelDTO, String componentId,
                                              ComponentIdentifier identifier, String modelName,
                                              ComponentAstDTOModelType modelType, List<ComponentParameterAstDTO> parameters,
                                              List<ComponentAstDTO> directChildren) {
    ComponentAstDTO componentAstDTO =
        new ComponentAstDTO(directChildren, UNKNOWN.toString(), identifier, mock(ComponentLocation.class), null, parameters,
                            componentId, dummyExtensionModelDTO, modelName, modelType, null);

    componentAstDTO.setGenerationInformation(new ComponentGenerationInformationDTO(new DslElementSyntax(identifier
        .getName(), identifier.getName(), "dum", DUMMY_NAMESPACE, false, false, false, false, false, null, new HashMap<>())));

    return componentAstDTO;
  }

  private ComponentMetadataAstDTO makeEmptyComponentMetadataAstDTO() {
    return new ComponentMetadataAstDTO(null, null, null, null, null, emptyList(), null, null, null, null);
  }

  private ParameterGroupModel mockParameterGroupModel(String paramGroupModelName) {
    ParameterGroupModel parameterGroupModel = mock(ParameterGroupModel.class);
    when(parameterGroupModel.getName()).thenReturn(paramGroupModelName);
    return parameterGroupModel;
  }

  private ParameterModel mockParameterModel(String paramModelName, ParameterRole parameterRole, MetadataType metadataType) {
    ParameterModel parameterModel = mock(ParameterModel.class);
    when(parameterModel.getName()).thenReturn(paramModelName);
    when(parameterModel.getRole()).thenReturn(parameterRole);
    when(parameterModel.getType()).thenReturn(metadataType);
    when(parameterModel.getDslConfiguration()).thenReturn(ParameterDslConfiguration.getDefaultInstance());
    return parameterModel;
  }

  private static class TestMetadataType implements MetadataType {

    @Override
    public MetadataFormat getMetadataFormat() {
      return new MetadataFormat("label", "id", "mimetype1", "mimetype2");
    }

    @Override
    public <T extends TypeAnnotation> Optional<T> getAnnotation(Class<T> annotation) {
      return empty();
    }

    @Override
    public Set<TypeAnnotation> getAnnotations() {
      return singleton(new ExampleAnnotation("exampleValue"));
    }

    @Override
    public Optional<String> getDescription() {
      return of("TestMetadataTypeDescription");
    }

    @Override
    public void accept(MetadataTypeVisitor visitor) {

    }
  }

}
