/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal.serialization.dto;

import static org.mule.runtime.ast.api.error.ErrorTypeBuilder.builder;

import static java.util.Collections.unmodifiableList;

import org.mule.runtime.api.message.ErrorType;

import java.util.List;

/**
 * Represents a node in the tree structure that represents the hierarchy of {@link ErrorType}s.
 */
public class ErrorTypeHierarchyItemDTO {

  private final String identifier;

  private final String namespace;

  private final boolean internal;

  private final List<ErrorTypeHierarchyItemDTO> children;

  /**
   * Creates an {@link ErrorType} for the information in this object with the provided parent.
   *
   * @param parent the parent of the errorType to be created.
   * @return a new {@link ErrorType}.
   */
  public ErrorType toErrorType(ErrorType parent) {
    return builder()
        .identifier(identifier)
        .namespace(namespace)
        .parentErrorType(parent)
        .build();
  }

  public ErrorTypeHierarchyItemDTO(String identifier, String namespace, boolean internal,
                                   List<ErrorTypeHierarchyItemDTO> children) {
    this.identifier = identifier;
    this.namespace = namespace;
    this.internal = internal;
    this.children = unmodifiableList(children);
  }

  /**
   * Identifier of the error. Is the value that is meant to be used in the configuration.
   *
   * @return the string representation of the error.
   */
  public String getIdentifier() {
    return identifier;
  }

  /**
   * The namespace of the module where the error is defined. For instance, for runtime errors the namespace is core.
   *
   * @return namespace of the module that defines this error.
   */
  public String getNamespace() {
    return namespace;
  }

  /**
   * @return {@code true} if the error represented by this object's {@link #getIdentifier()} and {@link #getNamespace()} is
   *         <b>NOT</b> handleable.
   */
  public boolean isInternal() {
    return internal;
  }

  /**
   * @return the error types that have the error represented by this object as parent.
   */
  public List<ErrorTypeHierarchyItemDTO> getChildren() {
    return children;
  }

}
