/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal.serialization.json.gson;

import static java.net.URI.create;
import static java.util.Collections.emptyList;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static java.util.Optional.ofNullable;
import static java.util.stream.Collectors.toList;
import static java.util.stream.StreamSupport.stream;

import org.mule.runtime.api.component.TypedComponentIdentifier;
import org.mule.runtime.api.component.location.ComponentLocation;
import org.mule.runtime.dsl.api.component.config.DefaultComponentLocation;
import org.mule.runtime.dsl.api.component.config.DefaultComponentLocation.DefaultLocationPart;

import java.lang.reflect.Type;
import java.net.URI;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.OptionalInt;
import java.util.stream.Collectors;
import java.util.stream.StreamSupport;

import com.google.gson.JsonArray;
import com.google.gson.JsonDeserializationContext;
import com.google.gson.JsonDeserializer;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParseException;

/**
 * This class implements a Gson {@link JsonDeserializer} and it can be registered to Gson as a Type Adapter in order to
 * deserialize {@link ComponentLocation}s
 */
public class ComponentLocationJsonDeserializer implements JsonDeserializer<ComponentLocation> {

  private TypedComponentIdentifierJsonDeserializer componentIdentifierJsonDeserializer =
      new TypedComponentIdentifierJsonDeserializer();

  @Override
  public ComponentLocation deserialize(JsonElement jsonElement, Type type, JsonDeserializationContext jsonDeserializationContext)
      throws JsonParseException {
    JsonObject locationJsonObject = jsonElement.getAsJsonObject();

    Optional<String> name = ofNullable(locationJsonObject.get("name"))
        .map(JsonElement::getAsString);

    List<DefaultLocationPart> parts = stream(locationJsonObject.get("parts").getAsJsonArray().spliterator(), false)
        .map(element -> deserializeLocationPart(element.getAsJsonObject(), jsonDeserializationContext))
        .collect(toList());

    List<URI> importChain;
    JsonElement importChainElement = locationJsonObject.get("importChain");
    if (importChainElement != null) {
      importChain = stream(importChainElement.getAsJsonArray().spliterator(), false)
          .map(element -> create(element.getAsString()))
          .collect(toList());
    } else {
      importChain = emptyList();
    }

    return new DefaultComponentLocation(name, parts, importChain);
  }

  public DefaultLocationPart deserializeLocationPart(JsonObject locationPartJsonObject,
                                                     JsonDeserializationContext jsonDeserializationContext) {
    JsonElement partPathJsonElement = locationPartJsonObject.get("partPath");
    String partPath = partPathJsonElement != null ? partPathJsonElement.getAsString() : null;

    Optional<TypedComponentIdentifier> partIdentifier =
        ofNullable(componentIdentifierJsonDeserializer.deserialize(locationPartJsonObject.get("partIdentifier"),
                                                                   TypedComponentIdentifier.class,
                                                                   jsonDeserializationContext));

    Optional<String> fileName = ofNullable(locationPartJsonObject.get("fileName"))
        .map(JsonElement::getAsString);

    OptionalInt lineInFile = OptionalInt.empty();
    JsonElement lineInFileJsonElement = locationPartJsonObject.get("line");
    if (lineInFileJsonElement != null) {
      lineInFile = OptionalInt.of(lineInFileJsonElement.getAsInt());
    }

    OptionalInt startColumn = OptionalInt.empty();
    JsonElement startColumnJsonElement = locationPartJsonObject.get("column");
    if (startColumnJsonElement != null) {
      startColumn = OptionalInt.of(startColumnJsonElement.getAsInt());
    }

    return new DefaultLocationPart(partPath, partIdentifier, fileName, lineInFile, startColumn);
  }
}
