/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal.serialization;

import static org.mule.runtime.ast.internal.serialization.json.JsonArtifactAstSerializerFormat.JSON;

import static java.lang.String.format;
import static java.util.Collections.singletonMap;
import static java.util.Objects.requireNonNull;
import static java.util.concurrent.Executors.newSingleThreadExecutor;
import static java.util.stream.Collectors.toList;

import static org.apache.commons.lang3.StringUtils.join;

import org.mule.runtime.ast.api.serialization.ArtifactAstDeserializer;
import org.mule.runtime.ast.api.serialization.ArtifactAstSerializer;
import org.mule.runtime.ast.internal.serialization.json.JsonArtifactAstSerializerFormat;

import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executor;
import java.util.stream.Stream;

/**
 * The serializer factory can create a {@link ArtifactAstSerializer} or a {@link ArtifactAstDeserializer} instance in the
 * available ids/formats
 */
public class ArtifactAstSerializerFactory {

  private final Executor executor;
  private final boolean populateGenerationInformation;
  private final Map<String, JsonArtifactAstSerializerFormat> formats =
      singletonMap(JSON, new JsonArtifactAstSerializerFormat());
  private final Map<String, ArtifactAstSerializer> versionToJsonSerializer = new ConcurrentHashMap<>();
  private final Map<String, InternalArtifactAstDeserializer> versionToJsonDeserializer = new ConcurrentHashMap<>();

  public ArtifactAstSerializerFactory() {
    this(newSingleThreadExecutor(), true);
  }

  public ArtifactAstSerializerFactory(Executor executor, boolean populateGenerationInformation) {
    requireNonNull(executor);
    this.executor = executor;
    this.populateGenerationInformation = populateGenerationInformation;
  }

  /**
   * This method creates a new instance of an {@link ArtifactAstSerializer} matching the format parameter
   *
   * @param formatId          the format in which the requested serializer serializes the
   *                          {@link org.mule.runtime.ast.api.ArtifactAst}
   * @param serializerVersion the desired serializer version
   * @return An {@link ArtifactAstSerializer} matching the format
   * @throws IllegalArgumentException if no {@link ArtifactAstSerializer} matching the format is found
   */
  public ArtifactAstSerializer getSerializer(String formatId, String serializerVersion)
      throws IllegalArgumentException {
    requireNonNull(formatId, "format");
    requireNonNull(serializerVersion, "serializerVersion");

    ArtifactAstSerializerFormat format = formats.get(formatId);
    if (formatIncludesVersion(format, serializerVersion)) {
      return versionToJsonSerializer.computeIfAbsent(serializerVersion, version -> format.createSerializer(version, executor));
    }

    throw new IllegalArgumentException(format("No serializer named '%s' version '%s' is available. Available serializers are: ["
        + join(getAvailableSerializers(), ", ") + "]", formatId, serializerVersion));
  }

  /**
   * This method creates a new instance o an {@link ArtifactAstDeserializer} matching the format and serializerVersion parameters
   *
   * @param formatId          the format which the requested deserializer should be able to deserialize in order to generate the
   *                          {@link org.mule.runtime.ast.api.ArtifactAst}
   * @param serializerVersion the desired version of the deserializer
   * @return An {@link ArtifactAstDeserializer} matching the format
   * @throws IllegalArgumentException if no {@link ArtifactAstDeserializer} matching the format is found
   */
  public InternalArtifactAstDeserializer getDeserializer(String formatId, String serializerVersion)
      throws IllegalArgumentException {
    requireNonNull(formatId, "format");
    requireNonNull(serializerVersion, "serializerVersion");

    ArtifactAstSerializerFormat format = formats.get(formatId);
    if (formatIncludesVersion(format, serializerVersion)) {
      return versionToJsonDeserializer
          .computeIfAbsent(serializerVersion, version -> format.createDeserializer(version, populateGenerationInformation));
    }

    throw new IllegalArgumentException(format("No deserializer named '%s' version '%s' is available. Available deserializers are: ["
        + join(getAvailableSerializers(), ", ") + "]", formatId, serializerVersion));
  }

  /**
   * All available serializer formats and versions. An available serializer format should be resolved by the
   * {@link #getSerializer(String, String)} and {@link #getDeserializer(String, String)} methods.
   *
   * @return a list of serializer formats and versions like: {@code "<formatId>[<version>]"}, e.g.: {@code "json[1.0]"}.
   */
  public List<String> getAvailableSerializers() {
    return formats.values()
        .stream()
        .flatMap(this::getAvailableSerializersByFormat)
        .collect(toList());
  }

  private Stream<String> getAvailableSerializersByFormat(ArtifactAstSerializerFormat format) {
    return format.getAvailableVersions()
        .stream()
        .map(version -> format("%s[%s]", format.getId(), version));
  }

  private static boolean formatIncludesVersion(ArtifactAstSerializerFormat format, String serializerVersion) {
    return format != null && format.getAvailableVersions().contains(serializerVersion);
  }
}
