/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal.serialization.json.gson;

import static org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType.SUB_FLOW;
import static org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType.valueOf;
import static org.mule.runtime.ast.internal.serialization.util.DslUtils.isSubflow;

import org.mule.runtime.api.component.ComponentIdentifier;
import org.mule.runtime.api.component.TypedComponentIdentifier;
import org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType;

import java.lang.reflect.Type;

import com.google.gson.JsonDeserializationContext;
import com.google.gson.JsonDeserializer;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParseException;

/**
 * This class implements a Gson {@link JsonDeserializer} and it can be registered to Gson as a Type Adapter in order to
 * deserialize {@link TypedComponentIdentifier}s
 *
 * @since 1.3
 */
public final class TypedComponentIdentifierJsonDeserializer implements JsonDeserializer<TypedComponentIdentifier> {

  private final ComponentIdentifierJsonDeserializer componentIdentifierJsonDeserializer =
      new ComponentIdentifierJsonDeserializer();

  @Override
  public TypedComponentIdentifier deserialize(JsonElement jsonElement, Type type,
                                              JsonDeserializationContext jsonDeserializationContext)
      throws JsonParseException {
    if (jsonElement == null) {
      return null;
    }

    JsonObject locationPartIdentifierJsonObject = jsonElement.getAsJsonObject();
    JsonElement identifier = locationPartIdentifierJsonObject.get("identifier");
    JsonElement componentType = locationPartIdentifierJsonObject.get("type");

    ComponentIdentifier componentIdentifier = identifier != null
        ? componentIdentifierJsonDeserializer.deserialize(identifier, ComponentIdentifier.class, jsonDeserializationContext)
        : null;
    return TypedComponentIdentifier.builder()
        .identifier(componentIdentifier)
        .type(getComponentType(componentType, componentIdentifier))
        .build();
  }

  private ComponentType getComponentType(JsonElement componentType, ComponentIdentifier componentIdentifier) {
    // There is an adaptation for legacy-serialized subflows here
    return isSubflow(componentIdentifier) ? SUB_FLOW : deserializeComponentType(componentType);
  }

  private ComponentType deserializeComponentType(JsonElement componentType) {
    return componentType != null ? valueOf(componentType.getAsJsonObject().get("name").getAsString()) : null;
  }
}
