/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.test.internal.serialization.dto.factory;

import static org.mule.runtime.ast.AllureConstants.ArtifactAstSerialization.AST_DTO;
import static org.mule.runtime.ast.AllureConstants.ArtifactAstSerialization.AST_SERIALIZATION;
import static org.mule.runtime.ast.internal.serialization.json.JsonArtifactAstSerializerFormat.JSON;

import static java.net.URI.create;
import static java.nio.charset.StandardCharsets.UTF_8;
import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static java.util.Collections.singletonList;
import static java.util.Collections.singletonMap;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.collection.IsCollectionWithSize.hasSize;

import org.mule.runtime.ast.api.ComponentMetadataAst;
import org.mule.runtime.ast.api.ImportedResource;
import org.mule.runtime.ast.internal.builder.PropertiesResolver;
import org.mule.runtime.ast.internal.serialization.ArtifactAstSerializerMetadata;
import org.mule.runtime.ast.internal.serialization.dto.ComponentMetadataAstDTO;
import org.mule.runtime.ast.internal.serialization.dto.ImportedResourceDTO;
import org.mule.runtime.ast.internal.serialization.dto.factory.AstDTOFactoryProvider;

import java.net.URI;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.OptionalInt;

import org.junit.Before;
import org.junit.Test;

import io.qameta.allure.Feature;
import io.qameta.allure.Story;

@Feature(AST_SERIALIZATION)
@Story(AST_DTO)
public class ComponentMetadataAstDTOFactoryTestCase {

  private AstDTOFactoryProvider astDTOFactoryProvider;
  private ComponentMetadataAst componentMetadataAst;
  private Optional<String> fileName;
  private Optional<URI> fileUri;
  private List<ImportedResource> importChain;
  private OptionalInt startLine;
  private OptionalInt startColumn;
  private OptionalInt endLine;
  private OptionalInt endColumn;
  private Optional<String> sourceCode;
  private Map<String, String> docAttributes;
  private Map<String, Object> parserAttributes;

  @Before
  public void setUp() throws Exception {
    astDTOFactoryProvider = new AstDTOFactoryProvider(new ArtifactAstSerializerMetadata(JSON, "1.0", UTF_8));
    fileName = empty();
    importChain = emptyList();
    fileUri = empty();
    startLine = OptionalInt.empty();
    startColumn = OptionalInt.empty();
    endLine = OptionalInt.empty();
    endColumn = OptionalInt.empty();
    sourceCode = empty();
    docAttributes = null;
    parserAttributes = null;

    componentMetadataAst = new TestComponentMetadataAst();
  }

  @Test
  public void testBuildReturnsMetadataDTOWithSameAttributesAsTheInput_WhenBuildingFromMetadataInputWithSetAttributes() {
    // Given
    this.fileName = of("some file name");
    URI rootConfigUri = create("file:///some/file/rootConfig.xml");
    ImportedResourceDTO importedResourceInChain = new ImportedResourceDTO("some file name",
                                                                          new ComponentMetadataAstDTO(null, null, null,
                                                                                                      "root config",
                                                                                                      rootConfigUri,
                                                                                                      emptyList(),
                                                                                                      null, null, null, null),
                                                                          null);
    importedResourceInChain.setPropertiesResolver(new PropertiesResolver());

    this.importChain = singletonList(importedResourceInChain);
    this.fileUri = of(create("file:///some/file/path.xml"));
    this.startLine = OptionalInt.of(4);
    this.startColumn = OptionalInt.of(2345);
    this.endLine = OptionalInt.of(96);
    this.endColumn = OptionalInt.of(3478);
    this.sourceCode = of("while(1){continue;}");
    this.docAttributes = new HashMap<>();
    this.docAttributes.put("onekey", "onevalue");
    this.docAttributes.put("twokey", "twovalue");
    this.parserAttributes = new HashMap<>();
    this.parserAttributes.put("somekey", new Object());
    this.parserAttributes.put("someotherkey", new Object());

    // When
    ComponentMetadataAstDTO componentMetadataAstDTO =
        astDTOFactoryProvider.getComponentMetadataAstDTOFactory().from(componentMetadataAst);
    componentMetadataAstDTO.enrich(singletonMap("some file name", importedResourceInChain));
    componentMetadataAstDTO.setPropertiesResolver(new PropertiesResolver());

    // Then
    assertThat(componentMetadataAstDTO.getFileName(), is(this.fileName));
    assertThat(componentMetadataAstDTO.getImportChain(), hasSize(1));
    ImportedResource importChain_0 = componentMetadataAstDTO.getImportChain().get(0);
    assertThat(importChain_0.getResourceLocation(), is("some file name"));
    assertThat(importChain_0.getMetadata().getFileName().get(), is("root config"));
    assertThat(importChain_0.getMetadata().getFileUri().get(), is(rootConfigUri));
    assertThat(componentMetadataAstDTO.getStartLine(), is(this.startLine));
    assertThat(componentMetadataAstDTO.getStartColumn(), is(this.startColumn));
    assertThat(componentMetadataAstDTO.getEndLine(), is(this.endLine));
    assertThat(componentMetadataAstDTO.getEndColumn(), is(this.endColumn));
    assertThat(componentMetadataAstDTO.getSourceCode(), is(this.sourceCode));
    assertThat(componentMetadataAstDTO.getDocAttributes(), is(this.docAttributes));
    assertThat(componentMetadataAstDTO.getParserAttributes(), is(this.parserAttributes));
  }

  @Test
  public void testBuildReturnsMetadataDTOWithSameAttributesAsTheInput_WhenBuildingFromMetadataInputWithNotSetAttributes() {
    // Given

    // When
    ComponentMetadataAstDTO componentMetadataAstDTO =
        astDTOFactoryProvider.getComponentMetadataAstDTOFactory().from(componentMetadataAst);

    componentMetadataAstDTO.enrich(emptyMap());

    // Then
    assertThat(componentMetadataAstDTO.getFileName(), is(this.fileName));
    assertThat(componentMetadataAstDTO.getImportChain(), is(this.importChain));
    assertThat(componentMetadataAstDTO.getStartLine(), is(this.startLine));
    assertThat(componentMetadataAstDTO.getStartColumn(), is(this.startColumn));
    assertThat(componentMetadataAstDTO.getEndLine(), is(this.endLine));
    assertThat(componentMetadataAstDTO.getEndColumn(), is(this.endColumn));
    assertThat(componentMetadataAstDTO.getSourceCode(), is(this.sourceCode));
    assertThat(componentMetadataAstDTO.getDocAttributes(), is(this.docAttributes));
    assertThat(componentMetadataAstDTO.getParserAttributes(), is(this.parserAttributes));
  }

  private class TestComponentMetadataAst implements ComponentMetadataAst {

    @Override
    public Optional<String> getFileName() {
      return fileName;
    }

    @Override
    public Optional<URI> getFileUri() {
      return fileUri;
    }

    @Override
    public List<ImportedResource> getImportChain() {
      return importChain;
    }

    @Override
    public OptionalInt getStartLine() {

      return startLine;
    }

    @Override
    public OptionalInt getStartColumn() {
      return startColumn;
    }

    @Override
    public OptionalInt getEndLine() {
      return endLine;
    }

    @Override
    public OptionalInt getEndColumn() {
      return endColumn;
    }

    @Override
    public Optional<String> getSourceCode() {
      return sourceCode;
    }

    @Override
    public Map<String, String> getDocAttributes() {
      return docAttributes;
    }

    @Override
    public Map<String, Object> getParserAttributes() {
      return parserAttributes;
    }
  }
}
