/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal.serialization.dto;

import static org.mule.runtime.ast.AllureConstants.ArtifactAstSerialization.AST_SERIALIZATION;
import static org.mule.runtime.ast.AllureConstants.ArtifactAstSerialization.AST_SERIALIZATION_ENRICH;

import static java.lang.Boolean.TRUE;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.not;
import static org.hamcrest.Matchers.nullValue;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

import org.mule.runtime.api.component.ComponentIdentifier;
import org.mule.runtime.api.exception.ErrorTypeRepository;
import org.mule.runtime.api.message.ErrorType;

import java.util.Collection;
import java.util.HashSet;
import java.util.Optional;
import java.util.Set;

import com.google.common.collect.ImmutableSet;
import io.qameta.allure.Feature;
import io.qameta.allure.Story;
import org.junit.Before;
import org.junit.Test;

@Feature(AST_SERIALIZATION)
@Story(AST_SERIALIZATION_ENRICH)
public class EnrichedErrorTypeRepositoryTestCase {

  private ErrorTypeRepository delegate;
  private EnrichedErrorTypeRepository repository;

  @Before
  public void setUp() {
    delegate = mock(ErrorTypeRepository.class);
    repository = new EnrichedErrorTypeRepository(delegate);
  }

  @Test
  public void testAddErrorTypeDelegatesIfPresent() {
    ComponentIdentifier identifier = mock(ComponentIdentifier.class);
    when(identifier.getName()).thenReturn("testIdentifer");
    when(identifier.getNamespace()).thenReturn("testNamespace");
    ErrorType parentError = mock(ErrorType.class);
    ErrorType existingError = mock(ErrorType.class);

    when(delegate.getErrorType(identifier)).thenReturn(Optional.of(existingError));

    ErrorType result = repository.addErrorType(identifier, parentError);

    assertThat(existingError, equalTo(result));
    verify(delegate).getErrorType(identifier);
  }

  @Test
  public void testAddErrorTypeCreatesNewIfAbsent() {
    ComponentIdentifier identifier = mock(ComponentIdentifier.class);
    ErrorType parentError = mock(ErrorType.class);

    when(delegate.getErrorType(identifier)).thenReturn(Optional.empty());
    when(identifier.getNamespace()).thenReturn("testNamespace");
    when(identifier.getName()).thenReturn("testName");

    ErrorType result = repository.addErrorType(identifier, parentError);

    assertThat(result, is(not(nullValue())));
    assertThat("testNamespace", equalTo(result.getNamespace()));
    assertThat("testName", equalTo(result.getIdentifier()));
  }

  @Test
  public void testLookupErrorTypeReturnsFromDelegateIfPresent() {
    ComponentIdentifier identifier = mock(ComponentIdentifier.class);
    ErrorType errorType = mock(ErrorType.class);

    when(delegate.lookupErrorType(identifier)).thenReturn(Optional.of(errorType));

    Optional<ErrorType> result = repository.lookupErrorType(identifier);

    assertThat(result.isPresent(), equalTo(TRUE));
    assertThat(errorType, equalTo(result.get()));
  }

  @Test
  public void testLookupErrorTypeReturnsFromLocalIfAbsentInDelegate() {
    ComponentIdentifier identifier = mock(ComponentIdentifier.class);
    ErrorType parentError = mock(ErrorType.class);

    when(delegate.lookupErrorType(identifier)).thenReturn(Optional.empty());
    when(identifier.getNamespace()).thenReturn("testNamespace");
    when(identifier.getName()).thenReturn("testName");

    repository.addErrorType(identifier, parentError);
    Optional<ErrorType> result = repository.lookupErrorType(identifier);

    assertThat(result.isPresent(), equalTo(TRUE));
    assertThat("testNamespace", equalTo(result.get().getNamespace()));
    assertThat("testName", equalTo(result.get().getIdentifier()));
  }

  @Test
  public void testGetErrorTypesIncludesDelegateAndLocal() {
    ErrorType delegateError = mock(ErrorType.class);
    ComponentIdentifier identifier = mock(ComponentIdentifier.class);
    ErrorType parentError = mock(ErrorType.class);

    when(delegate.getErrorTypes()).thenReturn(ImmutableSet.of(delegateError));
    when(identifier.getNamespace()).thenReturn("testNamespace");
    when(identifier.getName()).thenReturn("testName");

    repository.addErrorType(identifier, parentError);

    Set<ErrorType> result = repository.getErrorTypes();

    assertThat(result.contains(delegateError), equalTo(TRUE));
    assertThat(result.size(), equalTo(2)); // One from delegate, one from repository
  }

  @Test
  public void testGetInternalErrorTypesIncludesDelegateAndLocal() {
    ErrorType delegateError = mock(ErrorType.class);
    ComponentIdentifier identifier = mock(ComponentIdentifier.class);
    ErrorType parentError = mock(ErrorType.class);

    when(delegate.getInternalErrorTypes()).thenReturn(ImmutableSet.of(delegateError));
    when(identifier.getNamespace()).thenReturn("testNamespace");
    when(identifier.getName()).thenReturn("testName");

    repository.addInternalErrorType(identifier, parentError);

    Set<ErrorType> result = repository.getInternalErrorTypes();

    assertThat(result.contains(delegateError), equalTo(TRUE));
    assertThat(result.size(), equalTo(2)); // One from delegate, one from repository
    verify(delegate).getInternalErrorTypes();
  }

  @Test
  public void testGetErrorNamespacesCombinesNamespaces() {
    ErrorType error1 = mock(ErrorType.class);
    ErrorType error2 = mock(ErrorType.class);
    ErrorType internalError = mock(ErrorType.class);

    when(error1.getNamespace()).thenReturn("namespace1");
    when(error2.getNamespace()).thenReturn("namespace2");
    when(internalError.getNamespace()).thenReturn("internalNamespace");

    when(delegate.getErrorTypes()).thenReturn(ImmutableSet.of(error1, error2));
    when(delegate.getInternalErrorTypes()).thenReturn(ImmutableSet.of(internalError));

    Collection<String> namespaces = repository.getErrorNamespaces();

    assertThat(ImmutableSet.of("INTERNALNAMESPACE", "NAMESPACE2", "NAMESPACE1"), equalTo(new HashSet<>(namespaces)));
  }

  @Test
  public void testGetAnyErrorTypeDelegates() {
    ErrorType anyError = mock(ErrorType.class);
    when(delegate.getAnyErrorType()).thenReturn(anyError);

    assertThat(anyError, equalTo(repository.getAnyErrorType()));
    verify(delegate).getAnyErrorType();
  }

  @Test
  public void testGetSourceErrorTypeDelegates() {
    ErrorType sourceError = mock(ErrorType.class);
    when(delegate.getSourceErrorType()).thenReturn(sourceError);

    assertThat(sourceError, equalTo(repository.getSourceErrorType()));
    verify(delegate).getSourceErrorType();
  }

  @Test
  public void testGetSourceResponseErrorTypeDelegates() {
    ErrorType responseError = mock(ErrorType.class);
    when(delegate.getSourceResponseErrorType()).thenReturn(responseError);

    assertThat(responseError, equalTo(repository.getSourceResponseErrorType()));
    verify(delegate).getSourceResponseErrorType();
  }

  @Test
  public void testGetCriticalErrorTypeDelegates() {
    ErrorType criticalError = mock(ErrorType.class);
    when(delegate.getCriticalErrorType()).thenReturn(criticalError);

    assertThat(criticalError, equalTo(repository.getCriticalErrorType()));
    verify(delegate).getCriticalErrorType();
  }

  @Test
  public void testLookupErrorTypeDelegatesIfPresent() {
    ComponentIdentifier identifier = mock(ComponentIdentifier.class);
    ErrorType errorType = mock(ErrorType.class);

    when(delegate.lookupErrorType(identifier)).thenReturn(Optional.of(errorType));

    Optional<ErrorType> result = repository.lookupErrorType(identifier);

    assertThat(result.isPresent(), equalTo(TRUE));
    assertThat(errorType, equalTo(result.get()));
    verify(delegate).lookupErrorType(identifier);
  }

  @Test
  public void testGetErrorTypeDelegatesIfPresent() {
    ComponentIdentifier identifier = mock(ComponentIdentifier.class);
    ErrorType errorType = mock(ErrorType.class);

    when(delegate.getErrorType(identifier)).thenReturn(Optional.of(errorType));

    Optional<ErrorType> result = repository.getErrorType(identifier);

    assertThat(result.isPresent(), equalTo(TRUE));
    assertThat(errorType, equalTo(result.get()));
    verify(delegate).getErrorType(identifier);
  }
}
