/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast;

import static org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType.SUB_FLOW;
import static org.mule.runtime.api.meta.model.parameter.ParameterGroupModel.DEFAULT_GROUP_NAME;
import static org.mule.runtime.api.util.MuleSystemProperties.SYSTEM_PROPERTY_PREFIX;
import static org.mule.runtime.ast.AllureConstants.ArtifactAstSerialization.AST_SERIALIZATION;
import static org.mule.runtime.ast.AllureConstants.ArtifactAstSerialization.AST_SERIALIZATION_END_TO_END;
import static org.mule.runtime.ast.api.DependencyResolutionMode.MINIMAL;
import static org.mule.runtime.ast.internal.serialization.json.JsonArtifactAstSerializerFormat.JSON;
import static org.mule.tck.junit4.matcher.EitherMatcher.leftMatches;
import static org.mule.tck.junit4.matcher.EitherMatcher.rightMatches;

import static java.lang.System.clearProperty;
import static java.lang.System.setProperty;
import static java.util.stream.Collectors.toList;

import static com.google.common.collect.Streams.forEachPair;
import static org.apache.commons.io.FileUtils.copyInputStreamToFile;
import static org.hamcrest.CoreMatchers.equalTo;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.allOf;
import static org.hamcrest.Matchers.containsString;
import static org.hamcrest.Matchers.empty;
import static org.hamcrest.Matchers.endsWith;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.not;
import static org.hamcrest.Matchers.notNullValue;
import static org.hamcrest.Matchers.startsWith;
import static org.hamcrest.collection.IsCollectionWithSize.hasSize;
import static org.hamcrest.collection.IsIterableWithSize.iterableWithSize;
import static org.junit.Assert.assertThrows;

import org.mule.runtime.api.component.location.LocationPart;
import org.mule.runtime.api.functional.Either;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.meta.model.construct.ConstructModel;
import org.mule.runtime.api.meta.model.nested.NestableElementModel;
import org.mule.runtime.api.meta.model.parameter.ParameterModel;
import org.mule.runtime.ast.api.ArtifactAst;
import org.mule.runtime.ast.api.ComponentAst;
import org.mule.runtime.ast.api.ComponentMetadataAst;
import org.mule.runtime.ast.api.ComponentParameterAst;
import org.mule.runtime.ast.api.DependencyResolutionMode;
import org.mule.runtime.ast.api.ImportedResource;
import org.mule.runtime.ast.api.serialization.ArtifactAstDeserializer;
import org.mule.runtime.ast.api.serialization.ArtifactAstSerializer;
import org.mule.runtime.ast.api.serialization.ArtifactAstSerializerProvider;
import org.mule.runtime.ast.api.serialization.ExtensionModelResolver;
import org.mule.runtime.ast.internal.serialization.json.JsonArtifactAstSerializerFormat;
import org.mule.runtime.ast.testobjects.TestArtifactAstFactory;
import org.mule.runtime.ast.testobjects.TestExtensionModelResolver;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.math.BigDecimal;
import java.util.Iterator;
import java.util.List;
import java.util.Optional;
import java.util.OptionalInt;
import java.util.Set;
import java.util.stream.Stream;

import io.qameta.allure.Feature;
import io.qameta.allure.Issue;
import io.qameta.allure.Story;
import org.apache.commons.io.IOUtils;
import org.junit.After;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TemporaryFolder;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;
import org.junit.runners.Parameterized.Parameter;
import org.junit.runners.Parameterized.Parameters;

@Feature(AST_SERIALIZATION)
@Story(AST_SERIALIZATION_END_TO_END)
@RunWith(Parameterized.class)
public class ArtifactAstSerializationTestCase {

  @Parameters(name = "version: {0}")
  public static Iterable<String> data() {
    return new JsonArtifactAstSerializerFormat().getAvailableVersions();
  }

  @Rule
  public TemporaryFolder folder = new TemporaryFolder();

  @Parameter
  public String version;

  private ArtifactAstSerializerProvider artifactAstSerializerProvider;
  private TestArtifactAstFactory testArtifactAstFactory;
  private ExtensionModelResolver extensionModelResolver;
  private Set<ExtensionModel> extensionModels;

  @Before
  public void setUp() throws Exception {
    artifactAstSerializerProvider = new ArtifactAstSerializerProvider();
    testArtifactAstFactory = new TestArtifactAstFactory();
    extensionModels = testArtifactAstFactory.extensionModelsSetForTests();
    extensionModelResolver = new TestExtensionModelResolver(extensionModels);
  }

  @After
  public void tearDown() {
    clearProperty(SYSTEM_PROPERTY_PREFIX + DependencyResolutionMode.class.getName());
  }

  @Test
  public void testSerializationOfTrivialAppJsonImplementation() throws IOException {
    ArtifactAstSerializer artifactAstSerializer = artifactAstSerializerProvider.getSerializer(JSON, version);

    ArtifactAst trivialArtifactAst = testArtifactAstFactory.createArtifactFromXmlFile("test-apps/trivial.xml", extensionModels);

    InputStream serializedArtifactAst = artifactAstSerializer.serialize(trivialArtifactAst);

    String jsonString = IOUtils.toString(new InputStreamReader(serializedArtifactAst));
    assertThat(jsonString, containsString("JSON#" + version));
  }

  @Test
  public void testSerializationOfSimpleHttpAppJsonImplementation() throws IOException {
    ArtifactAstSerializer artifactAstSerializer = artifactAstSerializerProvider.getSerializer(JSON, version);

    ArtifactAst simpleHttpArtifactAst =
        testArtifactAstFactory.createArtifactFromXmlFile("test-apps/http-listener-with-logger.xml", extensionModels);

    InputStream serializedArtifactAst = artifactAstSerializer.serialize(simpleHttpArtifactAst);

    String jsonString = IOUtils.toString(new InputStreamReader(serializedArtifactAst));
    assertThat(jsonString, containsString("JSON#" + version));
    assertThat(jsonString, containsString("/test"));
    assertThat(jsonString, containsString("HTTP_Listener_config"));
    assertThat(jsonString, containsString("8081"));
    assertThat(jsonString, containsString("INFO"));
  }

  @Test
  public void testSerializationOfImportedSimpleHttpAppJsonImplementation() throws IOException {
    ArtifactAstSerializer artifactAstSerializer = artifactAstSerializerProvider.getSerializer(JSON, version);

    ArtifactAst simpleHttpArtifactAst =
        testArtifactAstFactory.createArtifactFromXmlFile("test-apps/imports-http-listener-with-logger.xml", extensionModels);

    InputStream serializedArtifactAst = artifactAstSerializer.serialize(simpleHttpArtifactAst);

    String jsonString = IOUtils.toString(new InputStreamReader(serializedArtifactAst));
    assertThat(jsonString, containsString("JSON#" + version));
    assertThat(jsonString, containsString("/test"));
    assertThat(jsonString, containsString("HTTP_Listener_config"));
    assertThat(jsonString, containsString("8081"));
    assertThat(jsonString, containsString("INFO"));
  }

  @Test
  public void testSerializationOfUnresolvableImportJsonImplementation() throws IOException {
    ArtifactAstSerializer artifactAstSerializer = artifactAstSerializerProvider.getSerializer(JSON, version);

    ArtifactAst simpleHttpArtifactAst =
        testArtifactAstFactory.createArtifactFromXmlFile("test-apps/failing-imports.xml", extensionModels);

    InputStream serializedArtifactAst = artifactAstSerializer.serialize(simpleHttpArtifactAst);

    String jsonString = IOUtils.toString(new InputStreamReader(serializedArtifactAst));
    assertThat(jsonString, containsString("JSON#" + version));
    assertThat(jsonString, containsString("Could not find imported resource \\u0027unresolvable.xml\\u0027"));
  }

  @Test
  public void testSerializationDeserializationOfTrivialAppJsonImplementation() throws IOException {
    ArtifactAstSerializer artifactAstSerializer = artifactAstSerializerProvider.getSerializer(JSON, version);

    ArtifactAst trivialArtifactAst = testArtifactAstFactory.createArtifactFromXmlFile("test-apps/trivial.xml", extensionModels);

    InputStream serializedArtifactAst = artifactAstSerializer.serialize(trivialArtifactAst);

    ArtifactAst artifactAst =
        artifactAstSerializerProvider.getDeserializer().deserialize(serializedArtifactAst, extensionModelResolver);

    assertThat(asIterable(trivialArtifactAst.topLevelComponentsStream()),
               iterableWithSize((int) artifactAst.topLevelComponentsStream().count()));

    assertThat(artifactAst.dependencies(), iterableWithSize(trivialArtifactAst.dependencies().size()));

    assertThat(artifactAst.getErrorTypeRepository().getErrorTypes().size(),
               is(trivialArtifactAst.getErrorTypeRepository().getErrorTypes().size()));
  }

  @Test
  public void testSerializationDeserializationOfSimpleHttpAppJsonImplementation() throws IOException {
    ArtifactAstSerializer artifactAstSerializer = artifactAstSerializerProvider.getSerializer(JSON, version);

    ArtifactAst simpleHttpArtifactAst =
        testArtifactAstFactory.createArtifactFromXmlFile("test-apps/http-listener-with-logger.xml", extensionModels);

    InputStream serializedArtifactAst = artifactAstSerializer.serialize(simpleHttpArtifactAst);

    ArtifactAstDeserializer deserializer = artifactAstSerializerProvider.getDeserializer();

    ArtifactAst artifactAst = deserializer.deserialize(serializedArtifactAst, extensionModelResolver);

    assertThat(asIterable(simpleHttpArtifactAst.topLevelComponentsStream()),
               iterableWithSize((int) artifactAst.topLevelComponentsStream().count()));

    assertThat(artifactAst.dependencies(), iterableWithSize(simpleHttpArtifactAst.dependencies().size()));

    List<ComponentParameterAst> componentParameterAstStream = artifactAst.topLevelComponentsStream()
        .flatMap(componentAst -> componentAst.getParameters().stream()).collect(toList());
    assertThat(componentParameterAstStream, not(empty()));

    // These assertions require the extension model to be properly formed.
    // A http extension model with a configuration model for the listener's config. This configuration model must have a
    // connection provider model for the listeners-connection and a source model for the http:listener message source
    // A mule extension model is also necessary for the flow construct and the logger operation.
    // Every one of these models needs to have its parameters defined as parameter models inside parameter groups.
    componentParameterAstStream
        .forEach(componentParameterAst -> assertThat(componentParameterAst.getModel(), notNullValue(ParameterModel.class)));
  }

  @Test
  public void testSerializationDeserializationOfSimpleHttpAppJsonImplementationKeepsComponentMetadataAstEquivalent()
      throws IOException {
    ArtifactAstSerializer artifactAstSerializer = artifactAstSerializerProvider.getSerializer(JSON, version);

    ArtifactAst simpleHttpArtifactAst =
        testArtifactAstFactory.createArtifactFromXmlFile("test-apps/http-listener-with-logger.xml", extensionModels);

    InputStream serializedArtifactAst = artifactAstSerializer.serialize(simpleHttpArtifactAst);

    ArtifactAstDeserializer deserializer = artifactAstSerializerProvider.getDeserializer();

    ArtifactAst artifactAst = deserializer.deserialize(serializedArtifactAst, extensionModelResolver);

    assertThat(asIterable(simpleHttpArtifactAst.topLevelComponentsStream()),
               iterableWithSize((int) artifactAst.topLevelComponentsStream().count()));

    // iterates all components recursively and asserts their metadata is equal
    forEachPair(artifactAst.recursiveStream().map(ComponentAst::getMetadata),
                simpleHttpArtifactAst.recursiveStream().map(ComponentAst::getMetadata),
                this::assertComponentMetadataAstEqual);
  }

  @Test
  @Issue("MULE-19775")
  public void testSerializationToFileAndDeserializationOfSimpleHttpAppJsonImplementationKeepsComponentMetadataAstEquivalent()
      throws IOException {
    ArtifactAstSerializer artifactAstSerializer = artifactAstSerializerProvider.getSerializer(JSON, version);

    ArtifactAst simpleHttpArtifactAst =
        testArtifactAstFactory.createArtifactFromXmlFile("test-apps/http-listener-with-logger.xml", extensionModels);

    InputStream serializedArtifactAst = artifactAstSerializer.serialize(simpleHttpArtifactAst);

    File file = folder.newFile("http-listener-artifact.ast");
    copyInputStreamToFile(serializedArtifactAst, file);

    ArtifactAstDeserializer deserializer = artifactAstSerializerProvider.getDeserializer();

    InputStream targetStream = new FileInputStream(file);

    ArtifactAst artifactAst = deserializer.deserialize(targetStream, extensionModelResolver);

    assertThat(asIterable(simpleHttpArtifactAst.topLevelComponentsStream()),
               iterableWithSize((int) artifactAst.topLevelComponentsStream().count()));

    // iterates all components recursively and asserts their metadata is equal
    forEachPair(artifactAst.recursiveStream().map(ComponentAst::getMetadata),
                simpleHttpArtifactAst.recursiveStream().map(ComponentAst::getMetadata),
                this::assertComponentMetadataAstEqual);
  }

  private void assertComponentMetadataAstEqual(ComponentMetadataAst actual, ComponentMetadataAst expected) {
    assertThat(actual.getDocAttributes(), equalTo(expected.getDocAttributes()));
    assertThat(actual.getParserAttributes(), equalTo(expected.getParserAttributes()));
    assertThat(actual.getFileName(), equalTo(expected.getFileName()));
    assertThat(actual.getFileUri(), equalTo(expected.getFileUri()));
    assertThat(actual.getSourceCode(), equalTo(expected.getSourceCode()));
    assertThat(actual.getStartColumn(), equalTo(expected.getStartColumn()));
    assertThat(actual.getStartLine(), equalTo(expected.getStartLine()));
    assertThat(actual.getEndColumn(), equalTo(expected.getEndColumn()));
    assertThat(actual.getEndLine(), equalTo(expected.getEndLine()));
  }


  @Test
  public void testSerializationDeserializationOfImportChain() throws IOException {
    ArtifactAstSerializer artifactAstSerializer = artifactAstSerializerProvider.getSerializer(JSON, version);

    ArtifactAst simpleHttpArtifactAst =
        testArtifactAstFactory.createArtifactFromXmlFile("test-apps/imports-http-listener-with-logger.xml", extensionModels);

    InputStream serializedArtifactAst = artifactAstSerializer.serialize(simpleHttpArtifactAst);

    ArtifactAstDeserializer deserializer = artifactAstSerializerProvider.getDeserializer();

    ArtifactAst artifactAst = deserializer.deserialize(serializedArtifactAst, extensionModelResolver);

    assertThat(asIterable(simpleHttpArtifactAst.topLevelComponentsStream()),
               iterableWithSize((int) artifactAst.topLevelComponentsStream().count()));

    assertThat(artifactAst.dependencies(), iterableWithSize(simpleHttpArtifactAst.dependencies().size()));

    List<ComponentAst> componentAstStream = artifactAst.topLevelComponents();

    componentAstStream
        .forEach(componentAst -> {
          List<ImportedResource> componentImportChain = componentAst.getMetadata().getImportChain();
          assertThat(componentImportChain, hasSize(1));
          assertThat(componentImportChain.get(0).getResourceLocation(), is("test-apps/http-listener-with-logger.xml"));
          assertThat(componentImportChain.get(0).getMetadata().getFileName().get(),
                     is("test-apps/imports-http-listener-with-logger.xml"));
          assertThat(componentImportChain.get(0).getMetadata().getFileUri().get().toString(),
                     allOf(startsWith("file:/"), endsWith("test-apps/imports-http-listener-with-logger.xml")));
        });
  }

  @Test
  public void testSerializationDeserializationOfAppWithDescriptions() throws IOException {
    ArtifactAstSerializer artifactAstSerializer = artifactAstSerializerProvider.getSerializer(JSON, version);

    ArtifactAst trivialArtifactAst =
        testArtifactAstFactory.createArtifactFromXmlFile("test-apps/with-descriptions.xml", extensionModels);

    InputStream serializedArtifactAst = artifactAstSerializer.serialize(trivialArtifactAst);

    ArtifactAst artifactAst =
        artifactAstSerializerProvider.getDeserializer().deserialize(serializedArtifactAst, extensionModelResolver);

    assertThat(asIterable(trivialArtifactAst.topLevelComponentsStream()),
               iterableWithSize((int) artifactAst.topLevelComponentsStream().count()));

    assertThat(artifactAst.dependencies(), iterableWithSize(trivialArtifactAst.dependencies().size()));

    assertThat(artifactAst.getErrorTypeRepository().getErrorTypes().size(),
               is(trivialArtifactAst.getErrorTypeRepository().getErrorTypes().size()));
  }

  @Test
  @Issue("W-14722981")
  public void testSerializationDeserializationOfAppWithChoice() throws IOException {
    ArtifactAstSerializer artifactAstSerializer = artifactAstSerializerProvider.getSerializer(JSON, version);

    ArtifactAst artifactAst =
        testArtifactAstFactory.createArtifactFromXmlFile("test-apps/with-choice.xml", extensionModels);

    InputStream serializedArtifactAst = artifactAstSerializer.serialize(artifactAst);

    ArtifactAst deserializedArtifactAst =
        artifactAstSerializerProvider.getDeserializer().deserialize(serializedArtifactAst, extensionModelResolver);

    assertThat(asIterable(deserializedArtifactAst.topLevelComponentsStream()),
               iterableWithSize((int) artifactAst.topLevelComponentsStream().count()));

    assertThat(deserializedArtifactAst.dependencies(), iterableWithSize(artifactAst.dependencies().size()));

    assertThat(deserializedArtifactAst.getErrorTypeRepository().getErrorTypes(),
               iterableWithSize(artifactAst.getErrorTypeRepository().getErrorTypes().size()));

    assertChoiceWhenHasExpression(deserializedArtifactAst);
  }

  @Test
  @Issue("W-14722981")
  public void testDeserializationOfAppWithChoiceWithBrokenWhenExpressionValue() throws IOException {
    try (InputStream serializedArtifactAst =
        this.getClass().getClassLoader().getResourceAsStream("examples/with-choice-broken-when-value.json")) {
      ArtifactAst deserializedArtifactAst =
          artifactAstSerializerProvider.getDeserializer().deserialize(serializedArtifactAst, extensionModelResolver);
      assertChoiceWhenHasExpression(deserializedArtifactAst);
    }
  }

  private void assertChoiceWhenHasExpression(ArtifactAst artifactAst) {
    ComponentAst flow = artifactAst.topLevelComponents().get(0);
    ComponentAst choice = flow.directChildren().get(0);
    ComponentAst when = choice.directChildren().get(0);
    ComponentParameterAst expressionParameter = when.getParameter("General", "expression");
    Either<String, Boolean> expressionParameterValue = expressionParameter.getValue();
    assertThat(expressionParameterValue, leftMatches(is("true")));
  }

  @Test
  @Issue("MULE-19818")
  public void testSerializationDeserializationOfLocationFileCoordinates() throws IOException {
    ArtifactAstSerializer artifactAstSerializer = artifactAstSerializerProvider.getSerializer(JSON, version);

    ArtifactAst simpleHttpArtifactAst =
        testArtifactAstFactory.createArtifactFromXmlFile("test-apps/imports-http-listener-with-logger.xml", extensionModels);

    InputStream serializedArtifactAst = artifactAstSerializer.serialize(simpleHttpArtifactAst);

    ArtifactAstDeserializer deserializer = artifactAstSerializerProvider.getDeserializer();

    ArtifactAst artifactAst = deserializer.deserialize(serializedArtifactAst, extensionModelResolver);

    ComponentAst httpListenerConfig = artifactAst.topLevelComponents().get(0);
    assertThat(httpListenerConfig.getLocation().getLine().getAsInt(), is(6));
    assertThat(httpListenerConfig.getLocation().getColumn().getAsInt(), is(5));
    ComponentAst listenerConnection = httpListenerConfig.directChildren().get(0);
    assertThat(listenerConnection.getLocation().getLine().getAsInt(), is(7));
    assertThat(listenerConnection.getLocation().getColumn().getAsInt(), is(9));
    ComponentAst flow = artifactAst.topLevelComponents().get(1);
    assertThat(flow.getLocation().getLine().getAsInt(), is(9));
    assertThat(flow.getLocation().getColumn().getAsInt(), is(5));
    ComponentAst listenerSource = flow.directChildren().get(0);
    assertThat(listenerSource.getLocation().getLine().getAsInt(), is(10));
    assertThat(listenerSource.getLocation().getColumn().getAsInt(), is(9));
    ComponentAst logger = flow.directChildren().get(1);
    assertThat(logger.getLocation().getLine().getAsInt(), is(11));
    assertThat(logger.getLocation().getColumn().getAsInt(), is(9));
  }

  @Test
  @Issue("MULE-19826")
  public void testProcessorsPartPath() throws IOException {
    ArtifactAstSerializer artifactAstSerializer = artifactAstSerializerProvider.getSerializer(JSON, version);

    ArtifactAst simpleHttpArtifactAst =
        testArtifactAstFactory.createArtifactFromXmlFile("test-apps/imports-http-listener-with-logger.xml", extensionModels);

    InputStream serializedArtifactAst = artifactAstSerializer.serialize(simpleHttpArtifactAst);

    ArtifactAstDeserializer deserializer = artifactAstSerializerProvider.getDeserializer();

    ArtifactAst artifactAst = deserializer.deserialize(serializedArtifactAst, extensionModelResolver);

    ComponentAst flow = artifactAst.topLevelComponents().get(1);
    ComponentAst logger = flow.directChildren().get(1);

    LocationPart processorsLocationPart = logger.getLocation().getParts().get(1);
    assertThat(processorsLocationPart.getPartPath(), is("processors"));
    assertThat(processorsLocationPart.getPartIdentifier(), is(Optional.empty()));
    assertThat(processorsLocationPart.getFileName(), is(Optional.empty()));
    assertThat(processorsLocationPart.getLine(), is(OptionalInt.empty()));
    assertThat(processorsLocationPart.getColumn(), is(OptionalInt.empty()));
  }

  @Test
  @Issue("MULE-19850")
  public void serializeWithCustomRependencyResolutionModeFails() {
    setProperty(SYSTEM_PROPERTY_PREFIX + DependencyResolutionMode.class.getName(),
                MINIMAL.name());

    ArtifactAstSerializer artifactAstSerializer = artifactAstSerializerProvider.getSerializer(JSON, version);

    ArtifactAst trivialArtifactAst = testArtifactAstFactory.createArtifactFromXmlFile("test-apps/trivial.xml", extensionModels);

    assertThrows("Artifact AST serialization with `DependencyResolutionMode != COMPILE` would have limited usability when deserialized.",
                 IllegalStateException.class,
                 () -> artifactAstSerializer.serialize(trivialArtifactAst));

  }

  @Test
  @Issue("W-16094915")
  public void globalErrorHandlerRefSerializedAsConstruct() throws Exception {
    try (InputStream serializedArtifactAst =
        this.getClass().getClassLoader().getResourceAsStream("examples/global-error-hadler-ref-as-construct.json")) {
      ArtifactAst deserializedArtifactAst =
          artifactAstSerializerProvider.getDeserializer().deserialize(serializedArtifactAst, extensionModelResolver);

      final ComponentAst flowAst = deserializedArtifactAst.topLevelComponents().get(1);
      final ComponentAst errorHandlerRef = flowAst.directChildren().get(1);

      assertThat(errorHandlerRef.getModel(NestableElementModel.class).isPresent(), is(true));
      assertThat(errorHandlerRef.getModel(ConstructModel.class).isPresent(), is(false));
    }
  }

  @Test
  @Issue("W-16230302")
  public void subflowSerializedAsChain() throws Exception {
    assertSubflowIsProperlyTyped("examples/with-subflow-as-chain.json");
  }

  @Test
  @Issue("W-16230302")
  public void subflowSerializedAsScope() throws Exception {
    assertSubflowIsProperlyTyped("examples/with-subflow-as-scope.json");
  }

  @Test
  @Issue("W-16230302")
  public void subflowIsProperlyTypedAfterRoundTrip() throws Exception {
    ArtifactAstSerializer artifactAstSerializer = artifactAstSerializerProvider.getSerializer(JSON, version);

    ArtifactAst artifactAst = testArtifactAstFactory.createArtifactFromXmlFile("test-apps/with-subflow.xml", extensionModels);

    // Control test
    assertSubflowIsProperlyTyped(artifactAst);

    InputStream serializedArtifactAst = artifactAstSerializer.serialize(artifactAst);
    ArtifactAstDeserializer deserializer = artifactAstSerializerProvider.getDeserializer();
    assertSubflowIsProperlyTyped(deserializer.deserialize(serializedArtifactAst, extensionModelResolver));
  }

  @Test
  @Issue("W-16761707")
  public void bigDecimalRetainsPrecisionAfterRoundTrip() throws Exception {
    ArtifactAstSerializer artifactAstSerializer = artifactAstSerializerProvider.getSerializer(JSON, version);

    ArtifactAst artifactAst = testArtifactAstFactory.createArtifactFromXmlFile("test-apps/with-big-decimal.xml", extensionModels);

    // Control test
    assertBigDecimalParameterIs(artifactAst, "testConfigNoDecimals", new BigDecimal("10000"));
    assertBigDecimalParameterIs(artifactAst, "testConfigNotDouble", new BigDecimal("0.2"));

    InputStream serializedArtifactAst = artifactAstSerializer.serialize(artifactAst);
    ArtifactAstDeserializer deserializer = artifactAstSerializerProvider.getDeserializer();
    ArtifactAst deserializedArtifactAst = deserializer.deserialize(serializedArtifactAst, extensionModelResolver);
    assertBigDecimalParameterIs(deserializedArtifactAst, "testConfigNoDecimals", new BigDecimal("10000"));
    assertBigDecimalParameterIs(deserializedArtifactAst, "testConfigNotDouble", new BigDecimal("0.2"));
  }

  private void assertSubflowIsProperlyTyped(String serializedFilePath) throws IOException {
    try (InputStream serializedArtifactAst =
        this.getClass().getClassLoader().getResourceAsStream(serializedFilePath)) {
      ArtifactAst deserializedArtifactAst =
          artifactAstSerializerProvider.getDeserializer().deserialize(serializedArtifactAst, extensionModelResolver);

      assertSubflowIsProperlyTyped(deserializedArtifactAst);
    }
  }

  private void assertSubflowIsProperlyTyped(ArtifactAst artifactAst) {
    final ComponentAst subflowAst = artifactAst.topLevelComponents().get(1);
    final ComponentAst processorInsideSubflowAst = subflowAst.directChildren().get(0);

    assertThat(subflowAst.getComponentType(), is(SUB_FLOW));
    assertThat(subflowAst.getLocation().getComponentIdentifier().getType(), is(SUB_FLOW));
    assertThat(subflowAst.getLocation().getParts().get(0).getPartIdentifier().get().getType(), is(SUB_FLOW));

    assertThat(processorInsideSubflowAst.getLocation().getParts().get(0).getPartIdentifier().get().getType(), is(SUB_FLOW));
  }

  private Iterable<Object> asIterable(Stream<ComponentAst> stream) {
    return (() -> (Iterator) stream.iterator());
  }

  private void assertBigDecimalParameterIs(ArtifactAst artifactAst, String configName, BigDecimal expected) {
    ComponentAst testConfigAst = artifactAst.topLevelComponentsStream()
        .filter(c -> c.getComponentId().filter(configName::equals).isPresent())
        .findFirst().get();
    assertThat(testConfigAst.getParameter(DEFAULT_GROUP_NAME, "bigDecimalParameter").getValue(), rightMatches(is(expected)));
  }
}
