/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal.serialization.visitor;

import org.mule.metadata.api.model.ObjectType;
import org.mule.metadata.api.model.SimpleType;
import org.mule.metadata.api.visitor.MetadataTypeVisitor;
import org.mule.runtime.ast.api.model.ExtensionModelHelper;
import org.mule.runtime.ast.api.serialization.ExtensionModelResolver;
import org.mule.runtime.ast.internal.model.ParameterModelUtils;
import org.mule.runtime.ast.internal.serialization.dto.ComponentAstDTO;
import org.mule.runtime.ast.internal.serialization.dto.ComponentParameterAstDTO;
import org.mule.runtime.ast.internal.serialization.resolver.GenerationInformationResolver;

import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Visits SimpleTypes and uses them to resolve the models of a list of {@link ComponentAstDTO}s
 */
public class ArrayElementMetadataTypeVisitor extends MetadataTypeVisitor {

  private static final Logger LOGGER = LoggerFactory.getLogger(ArrayElementMetadataTypeVisitor.class);

  private final ExtensionModelHelper extensionModelHelper;
  private final GenerationInformationResolver generationInformationResolver;
  private final ExtensionModelResolver extensionModelResolver;
  private final ComponentParameterAstDTO ownerComponentParameterAst;
  private final ComponentAstDTO parentComponentAstDTO;
  private final ParameterModelUtils parameterModelUtils;

  public ArrayElementMetadataTypeVisitor(ComponentParameterAstDTO componentParameterAstDTO,
                                         ComponentAstDTO parentComponentAstDTO,
                                         ExtensionModelHelper extensionModelHelper,
                                         GenerationInformationResolver generationInformationResolver,
                                         ExtensionModelResolver extensionModelResolver,
                                         ParameterModelUtils parameterModelUtils) {
    this.extensionModelHelper = extensionModelHelper;
    this.ownerComponentParameterAst = componentParameterAstDTO;
    this.parentComponentAstDTO = parentComponentAstDTO;
    this.generationInformationResolver = generationInformationResolver;
    this.extensionModelResolver = extensionModelResolver;
    this.parameterModelUtils = parameterModelUtils;
  }

  @Override
  public void visitSimpleType(SimpleType simpleType) {
    LOGGER.debug("Enrichment: resolveParameterValueModel, value is list of Simple ComponentAstDTO, parameter {}",
                 ownerComponentParameterAst);
    List<ComponentAstDTO> componentAstDTOList = (List<ComponentAstDTO>) ownerComponentParameterAst.getValue().getRight();

    componentAstDTOList.forEach(componentAstDTO -> componentAstDTO
        .resolveSimpleTypeComponentModel(this.ownerComponentParameterAst, simpleType, this.extensionModelResolver,
                                         this.extensionModelHelper, this.generationInformationResolver,
                                         this.parameterModelUtils));
  }

  @Override
  public void visitObject(ObjectType objectType) {
    LOGGER.debug("Enrichment: resolveParameterValueModel, value is list of ComponentAstDTOs, parameter {}",
                 ownerComponentParameterAst);
    List componentAstDTOList = (List) ownerComponentParameterAst.getValue().getRight();

    componentAstDTOList
        .stream()
        // filter out error-mappings
        .filter(ComponentAstDTO.class::isInstance)
        .forEach(componentAstDTO -> ((ComponentAstDTO) componentAstDTO)
            .resolveModelsRecursively(parentComponentAstDTO, extensionModelHelper, extensionModelResolver,
                                      generationInformationResolver, parameterModelUtils));
  }
}
