/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal.serialization.json.gson;

import org.mule.runtime.api.component.location.ComponentLocation;
import org.mule.runtime.api.component.location.LocationPart;

import java.lang.reflect.Type;

import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonSerializationContext;
import com.google.gson.JsonSerializer;

/**
 * This class implements a Gson {@link JsonSerializer} and it can be registered to Gson as a Type Adapter in order to deserialize
 * {@link ComponentLocation}s
 */
public class ComponentLocationJsonSerializer implements JsonSerializer<ComponentLocation> {

  private final ComponentIdentifierJsonSerializer componentIdentifierJsonSerializer = new ComponentIdentifierJsonSerializer();

  @Override
  public JsonElement serialize(ComponentLocation location, Type typeOfSrc, JsonSerializationContext context) {
    JsonObject jsonLocation = new JsonObject();

    jsonLocation.addProperty("name", location.getParts().get(0).getPartPath());

    JsonArray parts = new JsonArray();
    jsonLocation.add("parts", parts);
    location.getParts().forEach(locationPart -> {
      JsonObject jsonLocationPart = serializeLocationPart(typeOfSrc, context, locationPart);
      parts.add(jsonLocationPart);
    });

    jsonLocation.addProperty("location", location.getLocation());
    return jsonLocation;
  }

  protected JsonObject serializeLocationPart(Type typeOfSrc, JsonSerializationContext context, LocationPart locationPart) {
    JsonObject jsonLocationPart = new JsonObject();

    jsonLocationPart.addProperty("partPath", locationPart.getPartPath());

    locationPart.getPartIdentifier()
        .ifPresent(partId -> {
          JsonObject partIdentifier = new JsonObject();

          partIdentifier.add("identifier",
                             componentIdentifierJsonSerializer.serialize(partId.getIdentifier(), typeOfSrc, context));
          JsonObject type = new JsonObject();
          type.addProperty("name", partId.getType().name());
          partIdentifier.add("type", type);

          jsonLocationPart.add("partIdentifier", partIdentifier);
        });

    jsonLocationPart.addProperty("fileName", locationPart.getFileName().orElse(null));
    locationPart.getLine().ifPresent(line -> jsonLocationPart.addProperty("line", line));
    locationPart.getColumn().ifPresent(column -> jsonLocationPart.addProperty("column", column));

    return jsonLocationPart;
  }
}
