/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal.serialization.json;

import static org.mule.runtime.ast.internal.serialization.json.JsonArtifactAstSerializerFormat.JSON;

import org.mule.runtime.api.component.ComponentIdentifier;
import org.mule.runtime.api.component.TypedComponentIdentifier;
import org.mule.runtime.api.component.location.ComponentLocation;
import org.mule.runtime.api.message.ErrorType;
import org.mule.runtime.ast.api.ArtifactAst;
import org.mule.runtime.ast.api.ComponentGenerationInformation;
import org.mule.runtime.ast.api.serialization.ExtensionModelResolver;
import org.mule.runtime.ast.internal.serialization.ArtifactAstSerializerMetadata;
import org.mule.runtime.ast.internal.serialization.InternalArtifactAstDeserializer;
import org.mule.runtime.ast.internal.serialization.dto.ArtifactAstDTO;
import org.mule.runtime.ast.internal.serialization.dto.ParserAttributesDTO;
import org.mule.runtime.ast.internal.serialization.json.gson.ArtifactAstSerializerMetadataAwareTypeAdapterFactory;
import org.mule.runtime.ast.internal.serialization.json.gson.ComponentIdentifierJsonDeserializer;
import org.mule.runtime.ast.internal.serialization.json.gson.ComponentLocationJsonDeserializer;
import org.mule.runtime.ast.internal.serialization.json.gson.ErrorTypeJsonDeserializer;
import org.mule.runtime.ast.internal.serialization.json.gson.ParserAttributesJsonTypeAdapter;
import org.mule.runtime.ast.internal.serialization.json.gson.PostProcessingEnabler;
import org.mule.runtime.ast.internal.serialization.json.gson.TypedComponentIdentifierJsonDeserializer;
import org.mule.runtime.ast.internal.serialization.resolver.DefaultGenerationInformationResolver;
import org.mule.runtime.ast.internal.serialization.resolver.GenerationInformationResolver;
import org.mule.runtime.ast.internal.serialization.resolver.NoOpGenerationInformationResolver;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import java.io.BufferedReader;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.charset.Charset;

/**
 * Json implementation of an {@link InternalArtifactAstDeserializer}. This deserializer assumes the {@link InputStream} has no
 * header as it should have been consumed by the
 * {@link org.mule.runtime.ast.internal.serialization.DefaultArtifactAstDeserializer} in order to decide to dispatch it here.
 */
public class JsonArtifactAstDeserializer implements InternalArtifactAstDeserializer {

  private final Gson gson;
  private final String version;
  private final GenerationInformationResolver generationInformationResolver;

  /**
   * This constructor allows a {@link GsonBuilder} parameter.
   * 
   * @param gsonBuilder                   a {@link GsonBuilder} to use as a starting point.
   * @param version                       the version this deserializer must adhere to.
   * @param populateGenerationInformation Whether to populate the {@link ComponentGenerationInformation}.
   */
  public JsonArtifactAstDeserializer(GsonBuilder gsonBuilder, String version, boolean populateGenerationInformation) {
    gsonBuilder.registerTypeAdapter(TypedComponentIdentifier.class, new TypedComponentIdentifierJsonDeserializer());
    gsonBuilder.registerTypeAdapter(ComponentIdentifier.class, new ComponentIdentifierJsonDeserializer());
    gsonBuilder.registerTypeAdapter(ComponentLocation.class, new ComponentLocationJsonDeserializer());
    gsonBuilder.registerTypeAdapter(ErrorType.class, new ErrorTypeJsonDeserializer());
    gsonBuilder.registerTypeAdapter(ParserAttributesDTO.class, new ParserAttributesJsonTypeAdapter());
    gsonBuilder.registerTypeAdapterFactory(new PostProcessingEnabler());

    gson = gsonBuilder.create();

    this.version = version;

    if (populateGenerationInformation) {
      this.generationInformationResolver = new DefaultGenerationInformationResolver();
    } else {
      this.generationInformationResolver = new NoOpGenerationInformationResolver();
    }
  }

  /**
   * This method takes an input stream of a serialized {@link ArtifactAst} with no metadata header and deserializes and enriches
   * it.
   *
   * @param artifactAstInputStream the input stream generated by an ArtifactAstSerializer. Max allowed size =
   *                               {@value MAX_BYTE_COUNT} bytes
   * @param extensionModelResolver an instance of {@link ExtensionModelResolver}
   * @return The deserialized {@link ArtifactAst}
   */
  @Override
  public ArtifactAst deserialize(InputStream artifactAstInputStream, Charset charset,
                                 ExtensionModelResolver extensionModelResolver, ArtifactAst parent) {
    // We need to delay this up to this point because we don't know the charset beforehand
    Gson gson = createGsonWithMetadata(new ArtifactAstSerializerMetadata(JSON, version, charset));

    ArtifactAstDTO artifactAstDTO =
        gson.fromJson(new BufferedReader(new InputStreamReader(artifactAstInputStream, charset)), ArtifactAstDTO.class);
    artifactAstDTO.setParent(parent);
    artifactAstDTO.enrich(extensionModelResolver, generationInformationResolver);
    return artifactAstDTO;
  }

  private Gson createGsonWithMetadata(ArtifactAstSerializerMetadata serializerMetadata) {
    return gson.newBuilder()
        .registerTypeAdapterFactory(new ArtifactAstSerializerMetadataAwareTypeAdapterFactory(serializerMetadata))
        .create();
  }
}
