/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.testobjects;

import static java.lang.Thread.currentThread;
import static java.util.Arrays.asList;
import static java.util.Collections.emptyList;
import static java.util.Collections.singletonList;

import static com.google.common.collect.ImmutableSet.of;

import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.meta.model.construct.ConstructModel;
import org.mule.runtime.api.meta.model.operation.OperationModel;
import org.mule.runtime.ast.api.ArtifactAst;
import org.mule.runtime.ast.api.xml.AstXmlParser;
import org.mule.runtime.core.api.extension.provider.MuleExtensionModelProvider;
import org.mule.runtime.dsl.api.xml.parser.ParsingPropertyResolver;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;

public class TestArtifactAstFactory {

  private final ArtifactAst parentArtifact;
  private final ParsingPropertyResolver parsingPropertyResolver;

  public TestArtifactAstFactory() {
    parentArtifact = null;
    parsingPropertyResolver = s -> s;
  }

  public ArtifactAst createArtifactFromXmlFile(String xmlResourcePath, Set<ExtensionModel> extensionModels) {
    AstXmlParser.Builder builder = new AstXmlParser.Builder();

    builder.withExtensionModels(extensionModels);

    builder.withParentArtifact(parentArtifact);
    builder.withPropertyResolver(parsingPropertyResolver);

    AstXmlParser xmlParser = builder.build();

    return xmlParser.parse(currentThread().getContextClassLoader().getResource(xmlResourcePath));
  }

  public Set<ExtensionModel> extensionModelsSetForTests() {
    ExtensionModel muleExtensionModel = MuleExtensionModelProvider.getExtensionModel();
    List<OperationModel> httpOperationModels = new ArrayList<>();
    DummyConnectionProviderModel httpListenerConnectionProviderModel =
        new DummyConnectionProviderModel("listener-connection",
                                         singletonList(new DummyParameterGroupModel("",
                                                                                    asList(new DummyParameterModel("host"),
                                                                                           new DummyParameterModel("port")))));
    DummyConfigurationModel httpListenerConfigurationModel =
        new DummyConfigurationModel("listener-config",
                                    singletonList(new DummyParameterGroupModel("",
                                                                               singletonList(new DummyParameterModel("name")))),
                                    emptyList(), emptyList(), httpListenerConnectionProviderModel);

    DummyConnectionProviderModel httpRequesterConnectionProviderModel =
        new DummyConnectionProviderModel("request-connection",
                                         singletonList(new DummyParameterGroupModel("",
                                                                                    asList(new DummyParameterModel("host"),
                                                                                           new DummyParameterModel("port")))));
    DummyConfigurationModel httpRequesterConfigurationModel =
        new DummyConfigurationModel("request-config",
                                    singletonList(new DummyParameterGroupModel("",
                                                                               asList(new DummyParameterModel("name"),
                                                                                      new DummyParameterModel("authentication")))),
                                    emptyList(), emptyList(), httpRequesterConnectionProviderModel);

    DummySourceModel httpListenerSourceModel =
        new DummySourceModel("listener",
                             singletonList(new DummyParameterGroupModel("", asList(new DummyParameterModel("path"),
                                                                                   new DummyParameterModel("config-ref")))));
    List<ConstructModel> httpConstructModels = new ArrayList<>();
    DummyExtensionModel httpExtensionModel =
        new DummyExtensionModel(
                                "http",
                                httpOperationModels,
                                asList(httpListenerConfigurationModel, httpRequesterConfigurationModel),
                                singletonList(httpListenerSourceModel),
                                httpConstructModels,
                                asList(httpListenerConnectionProviderModel, httpRequesterConnectionProviderModel));


    return of(new DummyExtensionModel("tls"), muleExtensionModel, httpExtensionModel, new DummyExtensionModel("db"),
              new DummyExtensionModel("file"), new DummyExtensionModel("wsc"), new DummyExtensionModel("os"));
  }
}
