/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.test.internal.serialization.dto;

import static org.mule.metadata.api.model.MetadataFormat.JAVA;
import static org.mule.runtime.api.meta.ExpressionSupport.SUPPORTED;
import static org.mule.runtime.ast.AllureConstants.ArtifactAstSerialization.AST_SERIALIZATION;
import static org.mule.runtime.ast.AllureConstants.ArtifactAstSerialization.AST_SERIALIZATION_ENRICH;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.core.Is.is;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

import org.mule.metadata.api.builder.BaseTypeBuilder;
import org.mule.runtime.api.functional.Either;
import org.mule.runtime.api.meta.model.parameter.ParameterGroupModel;
import org.mule.runtime.api.meta.model.parameter.ParameterModel;
import org.mule.runtime.ast.internal.builder.PropertiesResolver;
import org.mule.runtime.ast.internal.serialization.dto.ComponentMetadataAstDTO;
import org.mule.runtime.ast.internal.serialization.dto.ComponentParameterAstDTO;

import org.junit.Before;
import org.junit.Test;

import io.qameta.allure.Feature;
import io.qameta.allure.Issue;
import io.qameta.allure.Story;

@Feature(AST_SERIALIZATION)
@Story(AST_SERIALIZATION_ENRICH)
public class ComponentParameterAstDTOTestCase {

  private ParameterModel paramModel;
  private ParameterGroupModel paramGroupModel;

  @Before
  public void before() {
    paramModel = mock(ParameterModel.class);
    when(paramModel.getExpressionSupport()).thenReturn(SUPPORTED);
    paramGroupModel = mock(ParameterGroupModel.class);
  }

  @Test
  public void expressionWithPadding() {
    ComponentParameterAstDTO param =
        new ComponentParameterAstDTO(null, "    #[anExpression]\r\n", false, mock(ComponentMetadataAstDTO.class), "General",
                                     "expressionValue");
    param.setModel(paramModel, paramGroupModel);

    PropertiesResolver propertiesResolver = new PropertiesResolver();
    param.setPropertiesResolver(propertiesResolver);

    assertThat(param.getValue().getLeft(), is("anExpression"));
  }

  @Test
  @Issue("MULE-19822")
  public void expressionwithPropertyPlaceholders() {
    ComponentParameterAstDTO param =
        new ComponentParameterAstDTO(null, "#[${aProperty}]", false, mock(ComponentMetadataAstDTO.class), "General",
                                     "expressionValue");
    param.setModel(paramModel, paramGroupModel);

    PropertiesResolver propertiesResolver = new PropertiesResolver();
    param.setPropertiesResolver(propertiesResolver);
    propertiesResolver.setMappingFunction(p -> {
      if (p.equals("#[${aProperty}]")) {
        return "#[anExpression]";
      } else {
        return p;
      }
    });

    assertThat(param.getValue().getLeft(), is("anExpression"));
  }

  @Test
  @Issue("W-16602622")
  public void fixedValueWithPropertyPlaceholders() {
    ComponentParameterAstDTO param =
        new ComponentParameterAstDTO(null, "${sfdc-crm-sapi.http.request.config}", false, mock(ComponentMetadataAstDTO.class),
                                     "General",
                                     "expressionValue");
    when(paramModel.getType()).thenReturn(new BaseTypeBuilder(JAVA).objectType().build());
    param.setModel(paramModel, paramGroupModel);

    PropertiesResolver propertiesResolver = new PropertiesResolver();
    param.setPropertiesResolver(propertiesResolver);
    propertiesResolver.setMappingFunction(p -> {
      if (p.equals("${sfdc-crm-sapi.http.request.config}")) {
        return "sfdc-crm-sapi:HTTP_Request_configuration";
      } else {
        return p;
      }
    });

    assertThat(param.getValue().getRight(), is("sfdc-crm-sapi:HTTP_Request_configuration"));
  }

  @Test
  @Issue("W-15959903")
  public void propertyPlaceholderOnNumberParam() {
    ComponentParameterAstDTO param =
        new ComponentParameterAstDTO(null, "{aNumberProperty}", false, mock(ComponentMetadataAstDTO.class), "General",
                                     "numberValue");
    when(paramModel.getType()).thenReturn(new BaseTypeBuilder(JAVA).numberType().build());
    param.setModel(paramModel, paramGroupModel);

    PropertiesResolver propertiesResolver = new PropertiesResolver();
    param.setPropertiesResolver(propertiesResolver);
    propertiesResolver.setMappingFunction(p -> {
      if (p.equals("{aNumberProperty}")) {
        return "2";
      } else {
        return p;
      }
    });

    assertThat(param.getValue().getRight(), is(2));
  }
}
