/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.test.internal.serialization;

import static org.mule.runtime.ast.AllureConstants.ArtifactAstSerialization.AST_SERIALIZER_METADATA_SERIALIZATION;
import static org.mule.runtime.ast.internal.serialization.ArtifactAstSerializerMetadataSerializer.DELIMITER;

import static java.lang.System.lineSeparator;
import static java.nio.charset.StandardCharsets.ISO_8859_1;
import static java.nio.charset.StandardCharsets.UTF_8;

import static org.apache.commons.io.IOUtils.toInputStream;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.is;
import static org.junit.rules.ExpectedException.none;

import org.mule.runtime.ast.internal.serialization.ArtifactAstSerializerMetadata;
import org.mule.runtime.ast.internal.serialization.ArtifactAstSerializerMetadataSerializer;

import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.Charset;

import org.apache.commons.io.IOUtils;

import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;

import io.qameta.allure.Feature;
import io.qameta.allure.Issue;

@Feature(AST_SERIALIZER_METADATA_SERIALIZATION)
public class ArtifactAstSerializerMetadataSerializerTestCase {

  @Rule
  public ExpectedException exception = none();

  private ArtifactAstSerializerMetadataSerializer artifactAstSerializerMetadataSerializer;

  @Before
  public void setUp() {
    artifactAstSerializerMetadataSerializer = new ArtifactAstSerializerMetadataSerializer();
  }

  @Test
  public void testReadMetadataReturnsSerializedMetadataWithIDJsonVersion100_WhenReadingAnArtifactAstInputStreamSerializedWithJsonSerializer100()
      throws IOException {
    // Given
    String serializerId = "JSON";
    String serializerVersion = "1.0.0";
    String jsonBody = "{{{{{{{{{{}}}}}}}}}}";
    InputStream inputStream =
        toInputStream(serializerId + DELIMITER + serializerVersion + DELIMITER + UTF_8.name() + DELIMITER + lineSeparator()
            + jsonBody,
                      UTF_8);

    // When
    ArtifactAstSerializerMetadata artifactAstSerializerMetadata =
        artifactAstSerializerMetadataSerializer.readArtifactAstSerializerMetadataFromInputStream(inputStream);

    // Then
    assertThat(artifactAstSerializerMetadata.getSerializerId(), is(serializerId));
    assertThat(artifactAstSerializerMetadata.getSerializerVersion(), is(serializerVersion));
  }

  @Test
  public void testReadMetadataReturnsSerializedMetadataWithIDKlingonVersion123_WhenReadingAnArtifactAstInputStreamSerializedWithKlingonSerializer123()
      throws IOException {
    // Given
    String serializerId = "KLINGON";
    String serializerVersion = "1.2.3";
    String jsonBody = "{{{{{{{{{{}}}}}}}}}}";
    InputStream inputStream =
        toInputStream(serializerId + DELIMITER + serializerVersion + DELIMITER + UTF_8.name() + DELIMITER + lineSeparator()
            + jsonBody,
                      UTF_8);
    artifactAstSerializerMetadataSerializer = new ArtifactAstSerializerMetadataSerializer();

    // When
    ArtifactAstSerializerMetadata artifactAstSerializerMetadata =
        artifactAstSerializerMetadataSerializer.readArtifactAstSerializerMetadataFromInputStream(inputStream);

    // Then
    assertThat(artifactAstSerializerMetadata.getSerializerId(), is(serializerId));
    assertThat(artifactAstSerializerMetadata.getSerializerVersion(), is(serializerVersion));
  }

  @Test
  public void testReadMetadataThrowsIllegalArgumentException_WhenReadingAnArtifactAstInputStreamWithAnInvalidHeaderThatHasWayTooLongSerializerId()
      throws IOException {
    // Given
    // This Id's length is greater that 1002
    String serializerId =
        "ThisIdIsWayTooLongThisIdIsWayTooLongThisIdIsWayTooLongThisIdIsWayTooLongThisIdIsWayTooLongThisIdIsWayTooLongThisIdIsWayTooLongThisIdIsWayTooLongThisIdIsWayTooLongThisIdIsWayTooLongThisIdIsWayTooLongThisIdIsWayTooLongThisIdIsWayTooLongThisIdIsWayTooLongThisIdIsWayTooLongThisIdIsWayTooLongThisIdIsWayTooLongThisIdIsWayTooLongThisIdIsWayTooLongThisIdIsWayTooLongThisIdIsWayTooLongThisIdIsWayTooLongThisIdIsWayTooLongThisIdIsWayTooLongThisIdIsWayTooLongThisIdIsWayTooLongThisIdIsWayTooLongThisIdIsWayTooLongThisIdIsWayTooLongThisIdIsWayTooLongThisIdIsWayTooLongThisIdIsWayTooLongThisIdIsWayTooLongThisIdIsWayTooLongThisIdIsWayTooLongThisIdIsWayTooLongThisIdIsWayTooLongThisIdIsWayTooLongThisIdIsWayTooLongThisIdIsWayTooLongThisIdIsWayTooLongThisIdIsWayTooLongThisIdIsWayTooLongThisIdIsWayTooLongThisIdIsWayTooLongThisIdIsWayTooLongThisIdIsWayTooLongThisIdIsWayTooLongThisIdIsWayTooLongThisIdIsWayTooLongThisIdIsWayTooLongThisIdIsWayTooLongThisIdIsWayTooLongThisIdIsWayTooLongThisIdIsWayTooLongThisIdIsWayTooLongThisIdIsWayTooLongThisIdIsWayTooLongThisIdIsWayTooLong";
    String serializerVersion = "1.2.3";
    String jsonBody = "{{{{{{{{{{}}}}}}}}}}";
    InputStream inputStream =
        toInputStream(serializerId + DELIMITER + serializerVersion + DELIMITER + UTF_8.name() + DELIMITER
            + lineSeparator() + jsonBody,
                      UTF_8);
    artifactAstSerializerMetadataSerializer = new ArtifactAstSerializerMetadataSerializer();

    // Then
    exception.expect(IllegalArgumentException.class);
    exception.expectMessage("The serialized artifact input stream has an invalid header: ");

    // When
    artifactAstSerializerMetadataSerializer.readArtifactAstSerializerMetadataFromInputStream(inputStream);
  }

  @Test
  public void testReadMetadataThrowsIllegalArgumentException_WhenReadingAnArtifactAstInputStreamWithAnInvalidHeaderThatHasInvalidCharset()
      throws IOException {
    // Given
    // This Id's length is greater that 1002
    String serializerId =
        "JSON";
    String serializerVersion = "1.2.3";
    String jsonBody = "{{{{{{{{{{}}}}}}}}}}";
    InputStream inputStream =
        toInputStream(serializerId + DELIMITER + serializerVersion + DELIMITER + "DOGENCODING" + DELIMITER + lineSeparator()
            + jsonBody,
                      UTF_8);
    artifactAstSerializerMetadataSerializer = new ArtifactAstSerializerMetadataSerializer();

    // Then
    exception.expect(IllegalArgumentException.class);
    exception
        .expectMessage("The serialized artifact input stream has an invalid header: UnsupportedCharsetException: DOGENCODING");

    // When
    artifactAstSerializerMetadataSerializer.readArtifactAstSerializerMetadataFromInputStream(inputStream);
  }

  @Test
  public void testReadMetadataThrowsIllegalArgumentException_WhenReadingAnArtifactAstInputStreamWithAnInvalidHeaderThatHasNoDelimiters()
      throws IOException {
    // Given
    String serializerId = "SomeOtherId";
    String serializerVersion = "1.2.3";
    String jsonBody = "{{{{{{{{{{}}}}}}}}}}";
    InputStream inputStream =
        toInputStream(serializerId + serializerVersion + lineSeparator() + jsonBody, UTF_8);
    artifactAstSerializerMetadataSerializer = new ArtifactAstSerializerMetadataSerializer();

    // Then
    exception.expect(IllegalArgumentException.class);
    exception.expectMessage("The serialized artifact input stream has an invalid header: ");

    // When
    artifactAstSerializerMetadataSerializer.readArtifactAstSerializerMetadataFromInputStream(inputStream);
  }

  @Test
  public void testReadMetadataThrowsIllegalArgumentException_WhenReadingAnArtifactAstInputStreamWithNoHeader()
      throws IOException {
    // Given
    String jsonBody = "{{{{{{{{{{}}}}}}}}}}";
    InputStream inputStream = toInputStream(jsonBody, UTF_8);
    artifactAstSerializerMetadataSerializer = new ArtifactAstSerializerMetadataSerializer();

    // Then
    exception.expect(IllegalArgumentException.class);
    exception.expectMessage("The serialized artifact input stream has an invalid header: ");

    // When
    artifactAstSerializerMetadataSerializer.readArtifactAstSerializerMetadataFromInputStream(inputStream);
  }

  @Test
  public void testAddHeaderToInputStreamReturnsAnInputStreamWithTheHeaderLineSomeFormat124130FollowedByTheOriginalContent_WhenAddingAHeaderToAnInputStream()
      throws IOException {
    // Given
    String jsonBody = "{}";
    InputStream inputStream = toInputStream(jsonBody, UTF_8);
    String serializerId = "someformat";
    String serializerVersion = "12.41.30";
    Charset charset = UTF_8;
    ArtifactAstSerializerMetadata artifactAstSerializerMetadata =
        new ArtifactAstSerializerMetadata(serializerId, serializerVersion, charset);

    // When
    InputStream result =
        artifactAstSerializerMetadataSerializer.addArtifactAstSerializerMetadataToInputStream(inputStream,
                                                                                              artifactAstSerializerMetadata);

    // Then
    assertThat(IOUtils.toString(result, UTF_8),
               is(serializerId + "#" + serializerVersion + "#" + charset + "#\n" + jsonBody));
  }

  @Test
  public void testAddHeaderToInputStreamReturnsAnInputStreamWithTheHeaderLineJSON100FollowedByTheOriginalContent_WhenAddingAHeaderToAnInputStream()
      throws IOException {
    // Given
    String jsonBody = "{\"attribute\": \"Some string I guess\"}";
    InputStream inputStream = toInputStream(jsonBody, UTF_8);
    String serializerId = "JSON";
    String serializerVersion = "1.0.0";
    Charset charset = ISO_8859_1;
    ArtifactAstSerializerMetadata artifactAstSerializerMetadata =
        new ArtifactAstSerializerMetadata(serializerId, serializerVersion, charset);

    // When
    InputStream result =
        artifactAstSerializerMetadataSerializer.addArtifactAstSerializerMetadataToInputStream(inputStream,
                                                                                              artifactAstSerializerMetadata);

    // Then
    assertThat(IOUtils.toString(result, UTF_8),
               is(serializerId + "#" + serializerVersion + "#" + charset + "#\n" + jsonBody));
  }

  @Test
  public void testAddHeaderToInputStreamThrowsNPEInputStream_WhenAddingAHeaderToANullInputStream() {
    // Given
    InputStream inputStream = null;
    String serializerId = "JSON";
    String serializerVersion = "1.0.0";
    Charset charset = UTF_8;
    ArtifactAstSerializerMetadata artifactAstSerializerMetadata =
        new ArtifactAstSerializerMetadata(serializerId, serializerVersion, charset);

    // Then
    exception.expect(NullPointerException.class);
    exception.expectMessage("inputStream");

    // When
    artifactAstSerializerMetadataSerializer.addArtifactAstSerializerMetadataToInputStream(inputStream,
                                                                                          artifactAstSerializerMetadata);

  }

  @Test
  public void testAddHeaderToInputStreamThrowsNPEInputStream_WhenAddingANullSerializerMetadataHeaderToAnInputStream() {
    // Given
    String jsonBody = "{\"attribute\": \"Lorem ipsum or something\"}";
    InputStream inputStream = toInputStream(jsonBody, UTF_8);
    ArtifactAstSerializerMetadata artifactAstSerializerMetadata = null;

    // Then
    exception.expect(NullPointerException.class);
    exception.expectMessage("artifactAstSerializerMetadata");

    // When
    artifactAstSerializerMetadataSerializer.addArtifactAstSerializerMetadataToInputStream(inputStream,
                                                                                          artifactAstSerializerMetadata);
  }

  @Test
  @Issue("MULE-19805")
  public void deserializationIgnoresExtraFields() throws IOException {
    // Given
    String serializerId = "JSON";
    String serializerVersion = "1.0.0";
    String jsonBody = "{{{{{{{{{{}}}}}}}}}}";
    InputStream inputStream =
        toInputStream(serializerId + DELIMITER + serializerVersion + DELIMITER + UTF_8.name() + DELIMITER
        // yup, this is a field form the future!
            + "comeWithMeIfYouWantToLive" + DELIMITER + lineSeparator()
            + jsonBody,
                      UTF_8);

    // When
    ArtifactAstSerializerMetadata artifactAstSerializerMetadata =
        artifactAstSerializerMetadataSerializer.readArtifactAstSerializerMetadataFromInputStream(inputStream);

    // Then
    assertThat(artifactAstSerializerMetadata.getSerializerId(), is(serializerId));
    assertThat(artifactAstSerializerMetadata.getSerializerVersion(), is(serializerVersion));
  }
}
